/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/BitState.h>
#include <agxTerrain/PrimaryActiveZone.h>
#include <agxTerrain/SoilParticleAggregate.h>
#include <agxTerrain/SoilPenetrationResistance.h>
#include <agxTerrain/AggregateContactGenerator.h>
#include <agxTerrain/DeformController.h>
#include <agxTerrain/ShovelSettings.h>
#include <agxStream/Serializable.h>
#include <agx/Plane.h>

namespace agxCollide
{
  class Geometry;
  class Shape;
}

namespace agxTerrain
{
  //--- Forward declarations ---//
  class ShovelAggregateContactMaterialContainer;
  AGX_DECLARE_POINTER_TYPES(ShovelAggregateContactMaterialContainer);
  AGX_DECLARE_VECTOR_TYPES(ShovelAggregateContactMaterialContainer);
  class Terrain;
  class TerrainMaterial;

  using TerrainPtrVec = agx::VectorPOD<Terrain*>;

  AGX_DECLARE_POINTER_TYPES(Shovel);
  AGX_DECLARE_VECTOR_TYPES(Shovel);

  /**
  Shovel object used to interact with a terrain via an active zone that converts solid terrain to dynamic terrain which
  can be moved by the shovel rigid body.

  A shovel object consists of a rigid body, two edges and a cutting direction. A convex inner shape is created between
  the two edges, the top edge and the cutting edge, and the attached rigid body. When the cutting edge intersects a
  terrain surface an active zone is created in the cutting direction inside the terrain, which then converts solid mass
  into dynamic mass. The shovel receives feedback forces through a rigid body aggregate constructed from the intertia
  of the soil particles in the active zone.

  The shovel has 4 modes for deforming terrain: PRIMARY, DEFORM_BACK, DEFORM_LEFT and DEFORM_RIGHT.
  See the Shovel::ExcavationMode enum for details.
  */
  class AGXTERRAIN_EXPORT Shovel : public agxSDK::TerrainToolInstance
  {
  public:
    class ExcavationSettings;
    class Settings;

    /**
    Find first shovel with given name.
    \param simulation - simulation the shovel is part of
    \param name - name of the shovel
    \return shovel if found - otherwise nullptr
    */
    static Shovel* find(const agxSDK::Simulation* simulation, const agx::Name& name);

    /**
    Find shovel with given UUID.
    \param simulation - simulation the shovel is part of
    \param uuid - UUID of the shovel
    \return shovel if found - otherwise nullptr
    */
    static Shovel* find(const agxSDK::Simulation* simulation, const agx::Uuid& uuid);

    /**
    Finds all shovels in the given simulation.
    \param simulation - simulation with shovels.
    \return vector of shovels
    */
    static ShovelPtrVector findAll(const agxSDK::Simulation* simulation);

    /**
    Basic constructor.
    \param shovelBody - The base body of the shovel.
    \param topEdge - The top edge of the active zone, specified in the LOCAL shovelBody frame
                     that will be created in front of the shovel.
    \param cuttingEdge - The lowest edge on the shovel, specified in the LOCAL shovelBody frame
                         that serves as the cutting edge of the active zone.
    \param cuttingDirection - The cutting direction of the shovel where the penetration resistance will be active,
                              which is usually parallel to the lowest shovel plate that is used to initially
                              penetrate the soil.
    \param connectedShovel - Set if the shovel should be connected to another shovels InnerBody for purposes of
                             simulating composite shovels with varying shovel vectors in along the shovel geometry.
                             This means you can have multiple cutting edges with varying orientation per shovel where
                             you have a central shovel with a single inner shape. (Default: nullptr)
    */
    Shovel( agx::RigidBody* shovelBody,
            const agx::Line& topEdge,
            const agx::Line& cuttingEdge,
            const agx::Vec3& cuttingDirection,
            Shovel* connectedShovel=nullptr,
            ShovelDebugRenderSettings* settings = nullptr);

    /**
    Enum for describing the different excavation modes of a shovel:
    */
    enum class ExcavationMode : agx::UInt32
    {
      /**
      The primary mode of excavation, where the shovel digs along the cutting direction in the terrain horizontal plane.
      */
      PRIMARY = 0,
      /**
      The mode of excavation where the shovel is digging in the opposite direction
      of the cutting direction in the terrain horizontal plane. ( Example: Backside grading )
      */
      DEFORM_BACK = 1,
      /**
      The mode of excavation where the shovel is digging to the clock wise (Right) direction orthogonal
      to the cutting direction in the terrain horizontal plane. ( Example: Side push/grading )
      */
      DEFORM_RIGHT = 2,
      /**
      The mode of excavation where the shovel is digging to the counter-clock wise (Left) direction orthogonal
      to the cutting direction in the terrain horizontal plane. ( Example: Side push/grading )
      */
      DEFORM_LEFT = 3
    };

    /*
    \return if the shovel is valid after construction. If it is not valid, it will not be possible to add the shovel to
    a simulation.
    */
    virtual bool isValid() const override;

    /**
    \return the base rigid body from the shovel object.
    */
    agx::RigidBody* getRigidBody() const;

    /**
    \return the excavation settings for an excavation mode in the shovel.
    */
    ExcavationSettings& getExcavationSettings(ExcavationMode mode);

    /**
    \return the common settings for the shovel.
    */
    ShovelSettings* getSettings();

    /**
    \return the advanced settings for the shovel.
    */
    AdvancedShovelSettings* getAdvancedSettings();

    /**
    \returns the top edge in local space.
    */
    const agx::Line& getTopEdge() const;

    /**
    \returns the top edge in world space.
    */
    agx::Line getTopEdgeWorld() const;

    /**
    \returns the cutting edge in local space.
    */
    const agx::Line& getCuttingEdge() const;

    /**
    \returns the cutting edge in world space.
    */
    agx::Line getCuttingEdgeWorld() const;

    /**
    \returns the forward vector in local space.
    */
    const agx::Vec3& getCuttingDirection() const;

    /**
    \returns the forward vector in world space.
    */
    agx::Vec3 getCuttingDirectionWorld() const;

    struct AGXTERRAIN_EXPORT TeethSettings
    {
      agx::Real   toothLength;
      agx::Real   toothMinRadius;
      agx::Real   toothMaxRadius;
      agx::UInt   numberOfTeeth;
    };

    /**
    \return the computed cutting edge length of the Shovel.
    */
    agx::Real computeCuttingEdgeLength() const;

    /**
    Set a settings object for an ExcavationMode of the shovel. Excavation modes are as follows:
    - PRIMARY - The primary mode of excavation, where the shovel digs along the cutting
                direction in the terrain horizontal plane.
    - DEFORM_BACK  - The mode of excavation where the shovel is digging in the opposite
                     direction of the cutting direction in the terrain horizontal plane.
                     ( Example: Backside grading )
    - DEFORM_LEFT  - The mode of excavation where the shovel is digging to the counter-clock wise
                     (Left) direction orthogonal to the cutting direction in the terrain horizontal plane.
                     ( Example: Side push/grading )
    - DEFORM_RIGHT - The mode of excavation where the shovel is digging to the clock wise
                     (Right) direction orthogonal to the cutting direction in the terrain horizontal plane.
                     ( Example: Side push/grading )
    \param mode - The specified excavation mode that the settings will apply to.
    \param excavationSettings - The specified excavation settings that will apply to the excavation mode.
    */
    void setExcavationSettings( ExcavationMode mode, ExcavationSettings excavationSettings );

    /**
    Set the top edge in local space.
    \param topEdge - new top edge in local space.
    */
    void setTopEdge(agx::Line topEdge);

    /**
    Set the cutting edge in local space.
    \param cuttingEdge - new cutting edge in local space.
    */
    void setCuttingEdge(agx::Line cuttingEdge);

    /**
    \param cuttingDirection - new cutting direction in local space.
    */
    void setCuttingDirection(agx::Vec3 cuttingDirection);

    /**
    Set the cutting plane in local space.
    \param cutTopPlane - new cutting plane in local space.
    */
    void setCutTopPlane(agx::Plane cutTopPlane);

    /**
    \return cutting plane in local space. 
    */
    const agx::Plane& getCutTopPlane() const;

    /**
    \return cutting plane in world space.
    */
    agx::Plane getCutTopPlaneWorld() const;

    /**
    \return get current separation plane for each excavation mode
    */
    agx::Plane getCurrentSeparationPlane( ExcavationMode mode ) const;

    /**
    Get the soil penetration model used for the shovel. Available models are:
    - NO_MODEL - Only use the soil pressure above the cutting edge to determine the penetration resistance.
    - ELASTIC_PLASTIC_LIMIT - Calculate the expected penetration resistance based in the elastic plastic limit. (Default)
    - PLASTIC - Calculate the expected penetration resistance based on plastic expansion due to tooth pressure.

    \param model - The output variable where the penetration model that is used will be put, if successful.
    \return true if the model was successfully extracted, false otherwise.
    */
    bool getSoilPenetrationModel(agxTerrain::SoilPenetrationResistance::PenetrationModel& model) const;

    /**
    Set the soil penetration model used for the shovel. Available models are:

    - NO_MODEL - Only use the soil pressure above the cutting edge to determine the penetration resistance.
    - ELASTIC_PLASTIC_LIMIT - Calculate the expected penetration resistance based in the elastic plastic limit. (Default)
    - PLASTIC - Calculate the expected penetration resistance based on plastic expansion due to tooth pressure.

    \param model - The specified penetration model to use in the shovel against the terrain.
                   ( Default: ELASTIC_PLASTIC_LIMIT )
    \return true if the model was successfully changed, false otherwise.
    */
    bool setSoilPenetrationModel(agxTerrain::SoilPenetrationResistance::PenetrationModel model);

    /**
    \return the secondary cutting direction of the shovel
    */
    agx::Vec3 getSecondaryCuttingDirection() const;

    /**
    \return the secondary separation forward vector of the shovel
    */
    agx::Real getSecondarySeparationAngle() const;

    /**
    Get the last computed dead load fraction of the shovel, i.e how much of 
    it's inner volume that is filled with dynamic soil. The dead load fraction 
    ranges from 0.0 (empty), to 1.0 (full).
    \return the last computed dead load fraction
    */
    agx::Real getDeadLoadFraction() const;

    /**
    Get the last computed dead load bulk volume of the shovel, i.e the inner 
    volume that is filled with dynamic soil. This is the particle volume inside 
    the shovel divied by the packing fraction of the soil (0.67).
    \return the last computed dead load bulk volume of the shovel
    */
    agx::Real getSoilVolume() const;

    /**
    \return the specific particle volume inside the shovel, i.e the soil volume 
    inside the shovel that is unmodified by the packing fraction of the soil.
    */
    agx::Real getParticleVolume() const;

    /**
    \return the shovel inner shape volume that is used in the dead load calculations
    */
    agx::Real getInnerVolume() const;

    /**
    Get the last computed inner contact area of the shovel, i.e the estimated cross-section area
    of the inner volume that is filled with dynamic soil.
    \note - This is used in the shovel-aggregate contact in primary excavation to calculate stiffness and cohesion.
    \return the last computed inner contact area of the shovel.
    */
    agx::Real getInnerContactArea() const;

    /**
    Change state enable of this shovel. If the shovel is disabled, it can not be used for excavation in a terrain.
    Default: true.
    \param enable - true to enable, false to disable
    */
    void setEnable(bool enable);

    /**
    Access the state enable flag.
    \return true if the body is enabled (default) - otherwise false
    */
    bool getEnable() const;

    /**
    \return the ShovelAggregateContactMaterialContainer that contains all contact materials explicitly set for
            this shovel.
    */
    ShovelAggregateContactMaterialContainer* getShovelTerrainContactMaterialContainer() const;

    /**
    \return the shovel geometries colliding with voxels in the terrain
    */
    const agxCollide::GeometryRefVector& getVoxelCollisionGeometries() const;

  public:

    /**
    \return parent frame located at model center of the shovel
    */
    agx::Frame* getParentFrame() const;

    /**
    Check if any of the collision geometries are inside terrain bounds
    */
    bool collisionGeometriesAreInsideTerrain(Terrain* terrain) const;

    /**
    \return soil particle aggregate specific for this shovel instance
    */
    SoilParticleAggregate* getSoilParticleAggregate() const;

    /**
    \return shovel penetration resistance specific for this shovel instance
    */
    SoilPenetrationResistance* getPenetrationResistance() const;

    /**
    \return shovel active zone specific for this shovel instance
    */
    PrimaryActiveZone* getActiveZone() const;

    /**
    \return aggregate contact generator specific for this shovel instance
    */
    AggregateContactGenerator* getAggregateContactGenerator() const;

    /**
    \return aggregate contact generator specific for this shovel instance
    */
    DeformController* getDeformController() const;

    /**
    \return the common material used for the SoilParticleAggregates inside the primary active zone and deformers.
    */
    agx::Material* getAggregateMaterial() const;

    /**
    Set if regular geometry contacts should be created between the shovel
    geometries and the associated terrain geometry.
    */
    void setEnableShovelTerrainGeometryContacts(Terrain* terrain, bool enable);

    /**
    Set if the shovel is enabled for the particular terrain. Prerequisites: Shovel needs to be added to the simulation.
    Default: True, ie enabled for any terrain.
    */
    void setEnableForTerrain(Terrain* terrain, bool enable);

    /**
    \return if the shovel is enabled for the specified terrain. Default: true.
    */
    bool getEnableForTerrain(Terrain* terrain) const;


    /**
    Explicitly set contact material properties in a shovel-aggregate contact for a specific
    excavation mode for the shovel. This overrides the shovel-terrain contact
    material properties that are used in the default case.
    \param contactMaterial - The contact material to be set in the aggregate contact.
    \param mode - The specified excavation mode that corresponds to the aggregate.
    \return true if the contact material was successfully set, false otherwise.
    */
    bool setShovelAggregateContactMaterial( agx::ContactMaterial* contactMaterial,
                                            Shovel::ExcavationMode mode = Shovel::ExcavationMode::PRIMARY );


    /**
    Get the explicitly set contact material in a shovel-aggregate contact corresponding to a specified
    excavation mode for the shovel. This overrides the shovel-terrain contact
    material properties that are used in the default case.
    \param mode - The specified excavation mode that corresponds to the aggregate.
    \note - this returns nullptr if the shovel is not valid.
    \return a contact material if one has been explicitly set, nullptr otherwise.
    */
    agx::ContactMaterial* getShovelAggregateContactMaterial( Shovel::ExcavationMode mode = Shovel::ExcavationMode::PRIMARY ) const;


    /**
    \return the current wedge aggregate in the specified collection given an excavation mode if it exists, nullptr otherwise.
    */
    const agx::RigidBody* getWedgeAggregate(Shovel::ExcavationMode excavationMode) const;

    /**
    \return the current wedge aggregate depth in the specified collection given an excavation mode.
    */
    agx::Real getAggregateTerrainContactDepth(Shovel::ExcavationMode excavationMode) const;

    /**
    Get the shovels forbidden bounds.
    */
    const agx::Vector<agxCollide::BoundingAABB>& getForbiddenBounds();

    /**
    Get the shovel <-> aggregate contacts with the soil particle aggregate for the given excavation mode.
    \param excavationMode - the excavation mode that the soil aggregate belongs too ( PRIMARY, DEFORM_BACK,
                              DEFORM_RIGHT, DEFORM_LEFT )
    \return vector containing the geometry contacts between the specified shovel and the aggregate.
    */
    agxCollide::GeometryContactPtrVector getShovelAggregateContacts(Shovel::ExcavationMode excavationMode) const;

    /**
    Get the aggregate <-> terrain contact area given an excavation mode and a shovel.
    \param excavationMode - the excavation mode that the aggregate belongs too ( PRIMARY, DEFORM_BACK, DEFORM_RIGHT, DEFORM_LEFT )
    \return the aggregate <-> terrain contact area given an excavation mode and a shovel.
    */
    agx::Real getAggregateTerrainContactArea(Shovel::ExcavationMode excavationMode) const;

    /**
    Given geometry contacts exists and the solver has solved them - calculates total contact force between the terrain
    soil particle aggregate and the shovel. This represents the separation force that is required to move the excavated
    soil in the shovel active zone.
    \return the total contact force between given shovel and the soil particle aggregate in the terrain.
    */
    agx::Vec3 getSeparationContactForce() const;

    /**
    Given geometry contacts exists and the solver has solved them - calculates total contact normal force between the
    terrain soil particle aggregate and the shovel.
    \return the total contact normal force between given shovel and the soil particle aggregate in the terrain.
    */
    agx::Vec3 getSeparationNormalForce() const;

    /**
    Given geometry contacts exists and the solver has solved them - calculates total contact friction force between the
    terrain soil particle aggregate and the given shovel.
    \return the total contact friction force between given shovel and the soil particle aggregate in the terrain.
    */
    agx::Vec3 getSeparationFrictionForce() const;

    /**
    Given geometry contacts exists and the solver has solved them - calculates total contact force between the terrain
    deformation soil aggregate and the shovel. This represents the deformation force that is required to move soil via
    shovel deformation instead of excavation, i.e not excavation or digging. Examples of this would be side movement
    and backwards grading of the soil.
    \return the total contact force between given shovel and the deformer soil aggregates in the terrain.
    */
    agx::Vec3 getDeformationContactForce() const;

    /**
    Given geometry contacts exists and the solver has solved them - calculates total shovel contact force between the
    shovel and it's active terrain. This is the contact force that prevents the shovel from falling through the terrain
    when not in excavation mode, where contact feedback is generated from the soil aggregates.
    \note - This method returns regular contact forces ONLY when no soil aggregates are present to generate excavation
            feedback!
    \return the total non-excavation contact force between this terrain and the given shovel.
    */
    agx::Vec3 getContactForce() const;

    /**
    Calculates total contact force between the soil aggregate associated with the specified excavation mode in the
    shovel. This represents the deformation force that is required to move soil via shovel deformation instead of
    excavation, i.e not excavation or digging. Examples of this would be side movement and backwards grading of the
    soil.
    \param excavationMode - the excavation mode that the specified soil aggregate belongs to
    \return the total contact force between the shovel and the soil aggregate specified by the excavation mode
    */
    agx::Vec3 getExcavationModeContactForce(Shovel::ExcavationMode excavationMode) const;

    /**
    The result includes the active force and torque from the penetration resistance from the active terrain on the shovel if the shovel
    is digging in the terrain
    \param force - the penetration force out parameter
    \param torque - the penetration torque out parameter
    \return true if resulting force and torque was written to \p force and \p torque - otherwise false
    */
    bool getPenetrationForce(agx::Vec3& force, agx::Vec3& torque) const;

    /**
    Get the terrain <-> aggregate contact force for the given excavation mode.
    \param excavationMode - the excavation mode that the aggregate belongs too ( PRIMARY, DEFORM_BACK, DEFORM_RIGHT, DEFORM_LEFT )
    \return the total force (N) acting on the aggregate in the terrain-aggregate contact
    */
    agx::Vec3 getAggregateTerrainContactForce(Shovel::ExcavationMode excavationMode) const;

    /**
    Get the terrain <-> aggregate normal force for the given excavation mode.
    \param excavationMode - the excavation mode that the aggregate belongs too ( PRIMARY, DEFORM_BACK, DEFORM_RIGHT, DEFORM_LEFT )
    \return the total force (N) acting on the aggregate in the terrain-aggregate contact
    */
    agx::Vec3 getAggregateTerrainNormalForce(Shovel::ExcavationMode excavationMode) const;

    /**
    Get the terrain <-> aggregate tangential force for the given excavation mode.
    \param excavationMode - the excavation mode that the aggregate belongs too ( PRIMARY, DEFORM_BACK, DEFORM_RIGHT, DEFORM_LEFT )
    \return the total force (N) acting on the aggregate in the terrain-aggregate contact
    */
    agx::Vec3 getAggregateTerrainTangentialForce(Shovel::ExcavationMode excavationMode) const;


    /**
    Get the aggregate <-> terrain geometry contacts with the shovel given an excavation mode.
    \param excavationMode - the excavation mode that the aggregate belongs too ( PRIMARY, DEFORM_BACK, DEFORM_RIGHT, DEFORM_LEFT )
    \return vector containing the geometry contacts between the specified aggregate and the terrain
    */
    agxCollide::GeometryContactPtrVector getAggregateTerrainContacts(Shovel::ExcavationMode excavationMode) const;

    /**
    Check if the shovel is currently in digging mode with a terrain, i.e if the cutting edge is submerged.
    \note Internally check if the shovel is acting on an active terrain or not, see getActiveTerrain.
    \return whether the Shovel is currently digging or not.
    */
    bool isDigging() const;

    /**
    \return dynamic mass in the shovel, including both particles and fluid mass
    */
    agx::Real getDynamicMass() const;

    /**
    Returns the total soil aggregate mass in the shovel for the specific excavation mode. This function can be used to
    extract the active mass that the shovel is trying to displace in the failure zones during digging and deformation.
    \param excavationMode - The excavation mode of the aggregate that will be used to extract the mass.
    \return the total aggregate mass, including inner shape if excavation mode is PRIMARY_EXCAVATION.
   */
    agx::Real getSoilAggregateMass(Shovel::ExcavationMode excavationMode) const;

    /*
    \return the active terrain that the shovel is acting on. If the shovel is currently not excavating, return nullptr.
    */
    Terrain* getActiveTerrain() const;

    /*
    \return if the shovel is currently active. The shovel is active either while interacting with a terrain, or if
    there are particels in the inner shape.
    */
    bool isActive() const;

    /**
    Class containing the settings for the different ExcavationModes for a shovel.
    */
    class AGXTERRAIN_EXPORT ExcavationSettings
    {
    public:
      /**
      Default constructor
      */
      ExcavationSettings();

      /**
      Constructor taking an internal state variable from another excavation setting object.
      \param state - The internal state data from another excavation setting object.
      */
      ExcavationSettings(agx::UInt32 state);

      /**
      Set whether the excavation mode should be enabled, creating dynamic mass and generating force feedback. Default: true.
      \param enable - true/false if the excavation mode should generate force feedback and create dynamic mass.
      */
      void setEnable(bool enable);

      /**
      Set true/false if the excavation mode should create dynamic mass. Default: true.
      \param enable - true if the excavation mode should create dynamic mass.
      */
      void setEnableCreateDynamicMass(bool enable);

      /**
      Set true/false if the excavation mode should generate force feedback from created aggregates. Default: true.
      \param enable - true if the excavation mode should generate force feedback from created aggregates.
      */
      void setEnableForceFeedback(bool enable);

      /**
      \return whether the excavation mode should be enabled, creating dynamic mass and generating force feedback. Default: true.
      */
      bool getEnable() const;

      /**
      \return whether the excavation mode should generate force feedback from created aggregates. Default: true.
      */
      bool getEnableForceFeedback() const;

      /**
      \return whether the excavation mode should create dynamic mass. Default: true.
      */
      bool getEnableCreateDynamicMass() const;

      /**
      \return the internal state variable for the shovel settings.
      */
      agx::UInt32 getState() const;

      void store( agxStream::OutputArchive& out ) const { m_excavationFlags.store( out ); }
      void restore( agxStream::InputArchive& in ) { m_excavationFlags.restore( in ); }

    private:
      enum StateFlags : agx::UInt32
      {
        ENABLED = 1 << 0,
        ENABLE_CREATE_DYNAMIC_MASS = 1 << 1,
        ENABLE_FORCE_FEEDBACK = 1 << 2
      };
      using Flags = agx::BitState<StateFlags, agx::UInt32>;
      Flags m_excavationFlags;
    };

    DOXYGEN_START_INTERNAL_BLOCK()
    /**
    Internal method to calculate the cutting plane and cache it so it can be 
    retrieved later with getCutTopPlane(). The cutting plane is defined with
    its normal pointing outwards from the shovel body. 
    \return cutting plane in local space. 
    */
    agx::Plane calculateCutTopPlane();

    /**
    Internal method to check that the top edge and cutting edge directions 
    are left->right, by comparing to a vector pointing towards the shovel 
    object CM. Flips them if they are the wrong direction. 
    */
    void setEdgesCodirectional();

    /**
    \return true if the Shovel could be used to deform the given terrain.
    */
    agx::Bool canDeformTerrain(Terrain* terrain) const;

    bool getShovelPlanes(agx::Plane& bottomPlane,
                         agx::Plane& backPlane,
                         agx::Plane& leftPlane,
                         agx::Plane& rightPlane) const;

    void removeTerrainShovelContacts(Terrain* terrain);

    /* Contact filtering for deformers */
    void removeTerrainShovelContactsDeformer(Terrain* terrain);

    bool isEdgeSubmerged(const agx::Line& edge, const Terrain* terrain) const;

    bool isEdgeSubmergedInSolid(const agx::Line& edge, const Terrain* terrain) const;

    bool isEdgeSubmergedInHeightfield(const agx::Line& edge, const agxCollide::HeightField* heightfield, const Terrain* terrain) const;

    agx::Real getSubmergedEdgeLength(const agx::Line& edge, const Terrain* terrain) const;

    agx::Vec3 getCuttingEdgeVelocity();

    void setInnerContactArea( agx::Real contactArea );

    /* Set whether the settings are dirty. Prompts the shovel to update internal classes due to changed settings. */
    void setSettingsAreDirty(bool enable);

    bool getSettingsAreDirty() const;

    Shovel* getConnectedShovel();

    agx::Physics::GranularBodySystem* getGranularBodySystem() const;

    virtual void onAssemblyAdd( agxSDK::Assembly* assembly ) override;

    /**
    Get the direction which is considered "up" for the shovel. This is equivalent to the opposite direction
    which gravity is acting in. If the shovel has not yet been added to a simulation, this is undefined.
    \return the opposite direction of gravity if the shovel has been added to a simulation. If not, a zero vector.
    */
    agx::Vec3 getUpDirection() const;

    const agx::Vec3iVector& getVoxelsInActiveZoneLastTimeStep();

    ShovelDebugRenderSettings* getDebugRenderSettings() const { return m_debugRenderSettings; }

    AGXSTREAM_DECLARE_SERIALIZABLE( agxTerrain::Shovel );
    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    /**
    Default constructor used in serialization.
    */
    Shovel();

    /**
    Reference counted object - protected destructor.
    */
    virtual ~Shovel();

    /* Make sure cutting direction is normalized */
    agx::Vec3 ensureCuttingDirectionIsNormalized(agx::Vec3 cuttingDirection);

    /* Default value for contact region threshold */
    agx::Real computeDefaultContactRegionThreshold() const;

    /*
    Initialize shovel <-> terrain voxel collision geometries.
    */
    agxCollide::GeometryRefVector createVoxelCollisionGeometries(agx::Real tessellationLevel = 0.5) const;

    /*
    Creates a frame which uses the shovel body's frame as parent. Other shovel internals
    attaches to this frame to make sure they follow along the shovel rigid body.
    Requires that the shovel body isn't nullptr.
    */
    agx::Frame* createParentFrame();

    agxSDK::TerrainManager* getTerrainManager();

    TerrainPtrVec getTerrainsInSimulation();

    /**
    Add notification when a shovel is added to a simulation.
    */
    virtual void addNotification()  override;

    /**
    Remove notification when this shovel is removed from a simulation.
    */
    virtual void removeNotification() override;

    /**
    Set the common material used by the SoilParticleAggregates inside the primary active zone and deformers.
    Propagates the material into the internal classes.
    */
    void setAggregateMaterial(agx::Material* material);

    /**
    Initialize contact materials when added to simulation
    */
    void initalizeContactMaterialsForExistingTerrains(agxSDK::Simulation* simulation);

    /*
    Checks if all internal contact materials are correct.
    If there is an active terrain (ie, we are excavating or bulldozing)
    the contact material will be customized for that terrain.
    */
    void checkContactMaterials();

    /**
    Executes pre-collide events for agxTerrain::Shovel in the simulation. Shovel and Terrain step events are
    synchronized via the TerrainManager. In the preCollide step, we:
    - Check if the shovel is active either by colliding with a terrain or if there are particles
    in the shovel.
    - If we need to, we create soil wedges in the primary and secondary deformers. (onPreCollide calls)
    - Forbidden bounds around the shovel are calculated.
    */
    virtual void preCollide() override;

    /**
    Executes pre-step events for agxTerrain::Shovel in the simulation.
    Shovel and Terrain step events are synchronized via the TerrainManager.
    In the pre step we:
    - Filter contact points for the deformers
    - Calculate dead load fraction
    - Check and if need be, set the internal contact materials (this is a new ContactMaterial and transfer each pre step)
    - Run the onPre calls in all internal classes (aggregates, deformers, pen resistance)
    */
    virtual void pre() override;

    /**
    Executes post-step events for agxTerrain::Shovel in the simulation.
    Shovel and Terrain step events are synchronized via the TerrainManager.
    */
    virtual void post() override;

    /**
    Callback to be executed at the end of the time step
    Shovel and Terrain step events are synchronized via the TerrainManager.
    */
    virtual void last() override;

    void calculateDeadLoadFraction();

    /*
    Checks if there are particles inside the shovels collision geometries.
    */
    bool shovelHasParticles() const;

    /**
    Clear the active zone wedges of the primary active zones and deformers
    */
    void clearActiveZoneWedges() const;

    /**
    Calculate the shovels forbidden bounds around all the active zones.
    */
    void calculateForbiddenBounds();

    /**
    Cast from excavation mode enum to deformer collection index
    */
    static agx::UInt excavationModeToDeformerCollectionIndex(Shovel::ExcavationMode excavationMode);

  protected:
    ShovelDebugRenderSettingsRef                m_debugRenderSettings;

    struct SoilPenetrationParameters
    {
      agx::Real depthThreshold;
      agx::Real forceScaling;
      agx::Real maxForce;
    };

    // Configuration:
    agx::RigidBodyRef         m_shovelBody;
    agx::Line                 m_cuttingEdge;
    agx::Line                 m_topEdge;
    agx::Vec3                 m_cuttingDirection;
    agx::Plane                m_cutTopPlane;

    // Settings
    bool                      m_enabled;
    ShovelSettingsRef         m_shovelSettings;
    std::array< ExcavationSettings, 4 > m_excavationSettings;

    // Caching of dead load
    agx::Real                 m_deadLoadFraction;
    // Caching of inner contact area
    agx::Real                 m_innerContactArea;

    agx::FrameRef m_parentFrame; /* Parent frame for shovel geometries synchronized with model center of shovel body. */

    // Geometries used to find shovel-voxel collisions
    agxCollide::GeometryRefVector m_voxelCollisionGeometries;
    bool                          m_settingsAreDirty;

    Shovel*                       m_connectedShovel;

    ShovelAggregateContactMaterialContainerRef  m_shovelAggregateCMContainer;
    SoilParticleAggregateRef                    m_soilParticleAggregate;
    SoilPenetrationResistanceRef                m_penetrationResistance;
    // Voxel collision geometries HAVE to be calculated before creating the active zone otherwise it is not possible to create an inner shape.
    PrimaryActiveZoneRef                        m_activeZone;
    AggregateContactGeneratorRef                m_aggregateContactGenerator;
    DeformControllerRef                         m_deformController;

    agx::MaterialRef                            m_aggregateMaterial;
    bool                                        m_isActive;
    Terrain*                                    m_activeTerrain;
    agx::Vector<agxCollide::BoundingAABB>       m_forbiddenBounds;
    agx::Vec3iVector                            m_voxelsInPrimaryActiveZoneLastTimeStep;
    agxCollide::BoundingAABB                    m_collisionBound;
  };

  /**
  Simple container class for storing and handling explicit contact material for shovel-aggregate contacts,
  indexed by specified excavation mode.
  */
  class AGXTERRAIN_EXPORT ShovelAggregateContactMaterialContainer : public agx::Referenced, public virtual agxStream::Serializable
  {
  public:
    ShovelAggregateContactMaterialContainer();

    ~ShovelAggregateContactMaterialContainer();

    bool setContactMaterial(Shovel::ExcavationMode excavationMode, agx::ContactMaterial* contactMaterial);

    agx::ContactMaterial* getContactMaterial(Shovel::ExcavationMode excavationMode) const;

    DOXYGEN_START_INTERNAL_BLOCK()
    AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::ShovelAggregateContactMaterialContainer);
    DOXYGEN_END_INTERNAL_BLOCK()

  private:
    agx::Vector< agx::ContactMaterialRef > m_contactMaterials;
  };

  class AGXTERRAIN_EXPORT ShovelUtils
  {
    friend class Shovel;
  public:
    /*
    \return true if the given rigidBody is part of a shovel.
    */
    static bool isRigidBodyShovel(const agx::RigidBody* rigidBody);

    /*
    \return true if the given geometry is part of shovels rigidBody.
    */
    static bool isGeometryShovel(const agxCollide::Geometry* geometry);

  protected:
    /*
    Set some properties on the rigid body to mark it as part of a shovel.
    Used in the shovel constructor.
    */
    static void setIsRigidBodyShovel(agx::RigidBody* rigidBody, bool enable);
  };
}
