/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceRegistry.h>

#include <agxCollide/Shape.h>

namespace agxCollide
{
  class MeshData;
}

namespace agxSensor
{
  class RtInstanceData;

  struct RtSceneHandle;
  struct RtScene;
  struct RtShape;
  struct RtShapeInstanceHandle;
  struct RtMaterialHandle;

  using RtSceneHandleRef = std::shared_ptr<RtSceneHandle>;
  using RtSceneRef = std::shared_ptr<RtScene>;

  using RtShapeRef = std::shared_ptr<RtShape>;
  using RtShapeInstanceHandleRef = std::shared_ptr<RtShapeInstanceHandle>;
  using RtMaterialHandleRef = std::shared_ptr<RtMaterialHandle>;

  /**
  Material handle of implicit type in the raytrace environment.
  */
  struct AGXSENSOR_EXPORT RtMaterialHandle
  {
    enum class Type
    {
      UNKNOWN,
      OPAQUE_LAMBERTIAN,
      OPAQUE_GGX_AND_OREN_NAYAR,
      OPAQUE_BRDF_EXPLICIT,
      AMBIENT_ATMOSPHERE,
      NUM_TYPES
    };
  };

  /**
  Material instance in the raytrace environment, containing handle
  and type.
  */
  struct AGXSENSOR_EXPORT RtMaterialInstance
  {
    /**
    Create a new handle and instance in of a material of the given type
    in the raytrace environment.
    \param type - type of the material
    \return instance (check isValid()) of the material
    */
    static RtMaterialInstance create( RtMaterialHandle::Type type );

    /**
    "Cast" this material instance to the given type \p inType. This is
    short for checking the type while constructing materials given a
    RtMaterialInstance. If \p inType doesn't match the type of this
    instance, the returned instance will be invalid (with only default
    values). Returned instance will always have the type \p inType.
    \param inType - type to cast this instance to
    \return instance of the given type \p inType
    */
    RtMaterialInstance as( RtMaterialHandle::Type inType ) const;

    /**
    "Cast" this material instance the given template parameter Type.
    \tparam Type - material type to cast this instance to
    \return instance of the given type \p inType
    \sa as( RtMaterialHandle::Type )
    */
    template<RtMaterialHandle::Type Type>
    RtMaterialInstance as() const;

    /**
    Check if this material instance is of the given type \p inType.
    \param inType - material type
    \return true if \p inType matches the type of this instance, otherwise false
    */
    bool is( RtMaterialHandle::Type inType ) const;

    /**
    Check if this material instance is of the given type.
    \tparam Type - material type
    \return true if the type matches the type of this instance, otherwise false
    */
    template<RtMaterialHandle::Type Type>
    bool is() const;

    /**
    \return true of other.handle == this->handle && other.type == this->type, the
            latter to handle nullptr
    */
    bool operator== ( const RtMaterialInstance& other ) const;

    /**
    \return true of other.handle != this->handle || other.type != this->type
    */
    bool operator!= ( const RtMaterialInstance& other ) const;

    RtMaterialHandleRef handle{ nullptr };
    RtMaterialHandle::Type type{ RtMaterialHandle::Type::UNKNOWN };
  };

  inline RtMaterialInstance RtMaterialInstance::as( RtMaterialHandle::Type inType ) const
  {
    return is( inType ) ?
             *this :
             // Invalid material of given inType.
             RtMaterialInstance{ nullptr, inType };
  }

  template<RtMaterialHandle::Type Type>
  inline RtMaterialInstance RtMaterialInstance::as() const
  {
    return this->as( Type );
  }

  inline bool RtMaterialInstance::is( RtMaterialHandle::Type inType ) const
  {
    return this->type == inType;
  }

  template<RtMaterialHandle::Type Type>
  inline bool RtMaterialInstance::is() const
  {
    return this->is( Type );
  }

  inline bool RtMaterialInstance::operator== ( const RtMaterialInstance& other ) const
  {
    return handle == other.handle && type == other.type;
  }

  inline bool RtMaterialInstance::operator!= ( const RtMaterialInstance& other ) const
  {
    return !( *this == other );
  }

  /**
  A scene constitutes the raytrace environment and contains a number
  of raytrace instances.
  */
  struct AGXSENSOR_EXPORT RtSceneHandle
  {
    /**
    Create an empty raytracing environment scene.
    \return handle to the representation of the given data in the sensor environment
    */
    static RtSceneHandleRef create();
  };

  /**
  Raytrace scene with handle and ambient/atmospheric material.
  */
  struct AGXSENSOR_EXPORT RtScene
  {
    /**
    Create a new scene with default ambient/atmospheric material.
    */
    static RtSceneRef create();

    /**
    Assign ambient/atmosphere material.
    \param newMaterial - new material for the scene
    */
    void setMaterial( RtMaterialInstance newMaterial );

    RtSceneHandleRef handle{ nullptr };
    RtMaterialInstance material{ nullptr };
  };

  /**
  Handle and interface for shapes in the raytrace environment (alias RtShapeHandle).
  Instances (RtShapeInstance) references this shape.
  */
  struct AGXSENSOR_EXPORT RtShape
  {
    /**
    Create mesh handle given vertices and indices.
    \param vertices - vertices of the mesh
    \param indices - triangle indices of size 3 * number of vertices
    \return handle to the representation of the given data in the sensor environment
    */
    static RtShapeRef create( const agx::Vec3Vector& vertices, const agx::UInt32Vector& indices );

    /**
    Create mesh handle given vertices and indices.
    \param vertices - vertices of the mesh
    \param indices - triangle indices of size 3 * number of vertices
    \param normals - normals of the mesh
    \return handle to the representation of the given data in the sensor environment
    */
    static RtShapeRef create( const agx::Vec3Vector& vertices, const agx::UInt32Vector& indices, const agx::Vec3Vector& normals );

    /**
    Create mesh handle given collision mesh data.
    \param data - collision mesh data
    \return handle to the representation of the given data in the sensor environment
    */
    static RtShapeRef create( const agxCollide::MeshData* data );

    /**
    Create mesh handle given shape type. Note that the returned mesh handle can
    be a cached one for shape types that support scaling - box, cylinder and sphere.
    \param shapeType - primitive shape type
    \return handle to the representation of the given data in the sensor environment
    */
    static RtShapeRef create( agxCollide::Shape::Type shapeType );

    /**
    Create mesh handle given shape instance. Note that the returned mesh handle can
    be a cached one for shape types that support scaling - box, cylinder and sphere.
    \param shape - shape instance to create a mesh representation of
    \return handle to the representation of the given data in the sensor environment
    */
    static RtShapeRef create( const agxCollide::Shape* shape );

    /**
    Create mesh handle given the triangles in the given render data instance.
    \param renderData - render data instance to create a mesh representation of
    \return handle to the representation of the given data in the sensor environment
    */
    static RtShapeRef create( const agxCollide::RenderData* renderData );

    /**
    Update vertex positions of this mesh. Note that the number of vertices has to
    be identical to the previous number, which is unknown to this mesh handle.
    \param vertices - updated vertex positions
    */
    void setVertices( const agx::Vec3Vector& vertices );
  };
  using RtShapeHandle = RtShape;
  using RtShapeHandleRef = RtShapeRef;

  /**
  Handle to an instance in the raytrace environment.
  */
  struct AGXSENSOR_EXPORT RtShapeInstanceHandle
  {
    /**
    Update transform and/or scale of this instance.
    \param agxTransform - new transform
    \param scale - new mesh scale
    */
    void setTransform( const agx::AffineMatrix4x4& agxTransform,
                       agx::Vec3 scale = agx::Vec3( 1 ) );

    /**
    Assign id of this instance. This id is an optional field in RtOutput.
    \param id - id to set, NullRtEntityId will reset it to default
    */
    void setEntityId( RtEntityId id );

    /**
    Assign material to this instance. If invalid, the default material will
    be assigned.
    \param materialInstance - material to assign, default material if nullptr
    */
    void setMaterial( RtMaterialInstance materialInstance );
  };

  /**
  A node performs a single operation, e.g., raycast,
  gaussian blur, merging etc.
  */
  struct AGXSENSOR_EXPORT RtNode
  {
  };
}
