/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceSurfaceMaterial.h>

namespace agxStream
{
  class OutputArchive;
  class InputArchive;
}

namespace agxSensor
{
  /**
  Opaque surface material reflecting light according to the combined two-layer model with a 
  specular GGX microfacet top-layer and a second microfacet Oren-Nayar diffuse layer.

  This surface reflection model is suitable for rough surfaces with a glossy coating, such as 
  vehicle or machine paint, coated carbon fiber, reflective signs and some rock faces.
  */
  class AGXSENSOR_EXPORT RtGgxAndOrenNayarMaterial : public RtSurfaceMaterial
  {
    public:
      static constexpr RtMaterialHandle::Type Type = RtMaterialHandle::Type::OPAQUE_GGX_AND_OREN_NAYAR;

    public:
      static RtGgxAndOrenNayarMaterial create();

      /**
      \return the surface material associated with the given shape
      */
      static RtGgxAndOrenNayarMaterial get( const agxCollide::Shape* shape );

      /**
      \return the surface material associated with the given geometry
      */
      static RtGgxAndOrenNayarMaterial get( const agxCollide::Geometry* geometry );

      /**
      \return the surface material associated with the given rigid body
      */
      static RtGgxAndOrenNayarMaterial get( const agx::RigidBody* rb );

      /**
      \return the surface material associated with the given linked structure (agxCable::Cable, agxModel::Beam, agxVehicle::Track, ...)
      */
      static RtGgxAndOrenNayarMaterial get( const agxSDK::LinkedStructure* linkedStructure );

      /**
      \return the surface material associated with the given wire
      */
      static RtGgxAndOrenNayarMaterial get( const agxWire::Wire* wire );

      /**
      \return the surface material associated with the given terrain
      */
      static RtGgxAndOrenNayarMaterial get( const agxTerrain::Terrain* terrain );

      /**
      \return the surface material associated with the given terrain pager
      */
      static RtGgxAndOrenNayarMaterial get( const agxTerrain::TerrainPager* terrainPager );

    public:
      RtGgxAndOrenNayarMaterial( RtMaterialInstance instance );

      /**
      \return the real part of the material top-layer refractive index
      */
      float getRefractiveIndexReal() const;

      /**
      Assign new real part of the material top-layer refractive index. This the common refractive 
      index value of the specular top layer of the material.
      \param refractiveIndexReal - real part of the material refractive index
      */
      RtGgxAndOrenNayarMaterial setRefractiveIndexReal( float refractiveIndexReal );

      /**
      \return the imaginary part of the material top-layer refractive index
      */
      float getRefractiveIndexImaginary() const;

      /**
      Assign new imaginary part of the material top-layer refractive index. This component primarily 
      affects the absorption of light at the specular top layer of the material.
      \param refractiveIndexImaginary - imaginary part of the material refractive index
      */
      RtGgxAndOrenNayarMaterial setRefractiveIndexImaginary( float refractiveIndexImaginary);

      /**
      \return the Beckman roughness of the material top-layer 
      */
      float getBeckmanRoughness() const;

      /**
      Assign new Beckman roughness of the material top-layer. This value represents the root mean 
      square of the slopes of the top-layer specular microfacets.
      \param beckmanRoughness - Beckman roughness value
      \param linkOrenNayarRoughness - true if the second layer Oren-Nayar roughness should be 
             determined from the top-layer Beckman roughness given to this function
      */
      RtGgxAndOrenNayarMaterial setBeckmanRoughness( float beckmanRoughness, bool linkOrenNayarRoughness = false );

      /**
      \return the Oren-Nayar roughness of the diffuse second layer of this material
      */
      float getOrenNayarRoughness() const;

      /**
      Assign new Oren-Nayar roughness to the second layer of this material. This value represents the
      standard deviation of the normal distribution of the second layer microfacets.
      \param orenNayarRoughness - Oren-Nayar roughness value
      \param linkBeckmanRoughness - true if the top-layer Beckman roughness should be determined
             from the second layer Oren-Nayar roughness given to this function
      */
      RtGgxAndOrenNayarMaterial setOrenNayarRoughness( float orenNayarRoughness, bool linkBeckmanRoughness = false );

      /**
      \return the reflectivity of the diffuse second layer of this material
      */
      float getDiffuseReflectivity() const;

      /**
      Assign new reflectivity value to the second layer of this material. This value represents the
      maximum amount of light reflected from the diffuse second layer, after transmission through
      the specular top-layer.
      \param diffuseReflectivity - Reflectivity of the diffuse second layer, <= 0.97 for physical
      */
      RtGgxAndOrenNayarMaterial setDiffuseReflectivity( float diffuseReflectivity );

    public:
      void store( agxStream::OutputArchive& out ) const;
      void restore( agxStream::InputArchive& in );

    private:
      using RtSurfaceMaterial::RtSurfaceMaterial;
  };
}
