/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/LidarRayRange.h>
#include <agxSensor/LidarProperties.h>
#include <agxSensor/LidarRayPatternGenerator.h>

#include <agxSensor/RaytraceOutputNoise.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( LidarModel );
  AGX_DECLARE_POINTER_TYPES( LidarModelHorizontalSweep );
  AGX_DECLARE_POINTER_TYPES( LidarModelGeneric360HorizontalSweep10Hz );

  /**
  Base lidar model describing the fundamentals, such as ray pattern, range and beam
  properties, of a lidar.
  */
  class AGXSENSOR_EXPORT LidarModel : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Construct given ray pattern generator, range and (optional) properties.
      If \p properties is nullptr, new, default properties is created.
      \param rayPatternGenerator - specific ray pattern generator of this lidar model,
                                   invalid if nullptr
      \param range - defines minimum and maximum range of this lidar model
      \param properties - properties/parameters of this model
      \param outputNoises - list of noise output modifications
      */
      LidarModel( LidarRayPatternGenerator* rayPatternGenerator,
                  agx::RangeReal32 range,
                  LidarProperties* properties = nullptr,
                  RtOutputNoiseRefVector outputNoises = RtOutputNoiseRefVector() );

      /**
      \param printIssues - true to log warning if there are issues, false to be silent
      \return true if the lidar model has all components required to simulate a lidar
      */
      bool isValid( bool printIssues = true ) const;

      /**
      \return the ray pattern generator of this lidar model
      */
      LidarRayPatternGenerator* getRayPatternGenerator() const;

      /**
      \return the ray range of this lidar model
      */
      LidarRayRange* getRayRange() const;

      /**
      \return the properties/parameters of this lidar model
      */
      LidarProperties* getProperties() const;

      /**
      \return vector of output noises of this lidar model
      */
      const RtOutputNoiseRefVector& getOutputNoises() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~LidarModel() = default;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarModel );

    protected:
      LidarModel();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      LidarRayPatternGeneratorRef m_rayPatternGenerator;
      LidarRayRangeRef m_rayRange;
      LidarPropertiesRef m_properties;
      RtOutputNoiseRefVector m_outputNoises;
  };

  /**
  A lidar model defining a rotational lidar, scanning its surroundings by emitting racks of laser
  beams in a circular pattern around the lidar, based on a horizontal field of view.
  The laser beams in these racks will be angled vertically in a semi-circle shape based on a
  specified vertical field of view.
  */
  class AGXSENSOR_EXPORT LidarModelHorizontalSweep : public LidarModel
  {
    public:
      /**
      Constructs a horizontal sweep lidar based on the field of view, scanning resolution, frequency
      and additional optional lidar properties.
      \param fov  - field of view in radians x (horizontal) and y (vertical), centered about the lidar x-axis.
      \param resolution - resolution in radians x (horizontal) and y (vertical)
      \param frequency - number of revolutions per second
      \param properties - properties/parameters of this model
      \param outputNoises - list of noise output modifications
      */
      LidarModelHorizontalSweep( agx::Vec2 fov,
                                 agx::Vec2 resolution,
                                 agx::Real frequency,
                                 LidarProperties* properties = nullptr,
                                 RtOutputNoiseRefVector outputNoises = RtOutputNoiseRefVector() );

      DOXYGEN_START_INTERNAL_BLOCK()

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarModelHorizontalSweep );

    public:
      virtual ~LidarModelHorizontalSweep() = default;

    protected:
      LidarModelHorizontalSweep();

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  /**
  A simple 360 degree sweeping lidar model of a vertical field of view of +/- 50 degrees and
  a resolution of 0.5 degrees in both the vertical and horizontal axis.
  */
  class AGXSENSOR_EXPORT LidarModelGeneric360HorizontalSweep : public LidarModelHorizontalSweep
  {
    public:
      /**
      Constructs a 360 degree horizontally sweeping lidar model of the specified sweep frequency
      and additional optional lidar properties.
      \param frequency - number of revolutions per second
      \param properties - properties/parameters of this model
      \param outputNoises - list of noise output modifications
      */
      LidarModelGeneric360HorizontalSweep( agx::Real frequency,
                                           LidarProperties* properties = nullptr,
                                           RtOutputNoiseRefVector outputNoises = RtOutputNoiseRefVector() );

      DOXYGEN_START_INTERNAL_BLOCK()

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarModelGeneric360HorizontalSweep );

    public:
      virtual ~LidarModelGeneric360HorizontalSweep() = default;

    protected:
      LidarModelGeneric360HorizontalSweep();

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  /**
  A very simple short-hand for creating a basic 360 degree horizontally sweeping lidar locked at the
  common lidar scanning frequency of 10 Hz.
  */
  class AGXSENSOR_EXPORT LidarModelGeneric360HorizontalSweep10Hz :
    public LidarModelGeneric360HorizontalSweep
  {
    public:
      /**
      Construct a basic 360 degree horizontally sweeping lidar locked at 10 Hz.
      */
      LidarModelGeneric360HorizontalSweep10Hz();

      DOXYGEN_START_INTERNAL_BLOCK()

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarModelGeneric360HorizontalSweep10Hz );

    public:
      virtual ~LidarModelGeneric360HorizontalSweep10Hz() = default;

      DOXYGEN_END_INTERNAL_BLOCK()
  };
}
