/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxSensor/FrameAttachedSensor.h>
#include <agxSensor/LidarModel.h>
#include <agxSensor/LidarFrameNode.h>
#include <agxSensor/LidarRayDistortionHandler.h>
#include <agxSensor/RaytraceOutputHandler.h>

namespace agxSDK
{
  class Simulation;
}

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( Lidar );
  AGX_DECLARE_VECTOR_TYPES( Lidar );

  /**
  Lidar instance class defined by a frame/transform and a model. Any output,
  such as point cloud, intensity, normals etc., are given as seen from the
  lidar frame.
  */
  class AGXSENSOR_EXPORT Lidar : public FrameAttachedSensor
  {
    public:
      /**
      Find lidar given name.
      \param simulation - simulation with a sensor environment
      \param name - name of lidar instance to search for
      \return first lidar instance with name \p name if found, otherwise nullptr
      */
      static Lidar* find( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find lidar given name.
      \param environment - sensor environment
      \param name - name of lidar instance to search for
      \return first lidar instance with name \p name if found, otherwise nullptr
      */
      static Lidar* find( const Environment* environment, const agx::Name& name );

      /**
      Find all lidar given name.
      \param simulation - simulation with a sensor environment
      \param name - name of lidar instance to search for
      \return lidar instances with name \p name
      */
      static LidarPtrVector findAll( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find all lidar given name.
      \param environment - sensor environment
      \param name - name of lidar instance to search for
      \return lidar instances with name \p name
      */
      static LidarPtrVector findAll( const Environment* environment, const agx::Name& name );

      /**
      Find all lidar.
      \param simulation - simulation with a sensor environment
      \return all lidar instances in the sensor environment
      */
      static LidarPtrVector findAll( const agxSDK::Simulation* simulation );

      /**
      Find all lidar
      \param environment - sensor environment
      \return all lidar instances in the given sensor environment \p environment
      */
      static LidarPtrVector findAll( const Environment* environment );

    public:
      /**
      Construct this lidar given \p frame and \p model. If the given frame is nullptr,
      a new frame is created aligned with the world frame.
      \note This lidar instance is invalid if the given \p model is nullptr.
      \param frame - world frame transform of this lidar, may be dynamic, the output
                     will be given in this frame.
      \param model - model defining this lidar instance, invalid if nullptr
      */
      Lidar( agx::Frame* frame, LidarModel* model );

      /**
      \return the frame of this lidar, defining this lidar -> world transform.
      */
      virtual agx::Frame* getFrame() const override;

      /**
      Assign new frame to this lidar. The results will be given in this new frame the
      next time the environment is stepped.
      \param frame - new frame for this lidar
      */
      virtual void setFrame( agx::Frame* frame ) override;

      /**
      \return the frame node in the system tree updating the transform of this lidar
      */
      LidarFrameNode* getFrameNode() const;

      /**
      \return the model if this lidar
      */
      LidarModel* getModel() const;

      /**
      \return the ray distortion handler managing any potential distortion applied to the rays 
              before raytrace.
      */
      LidarRayDistortionHandler* getRayDistortionHandler() const;

      /**
      \return the output handler managing the outputs, such as point hits cloud, intensities,
              normals, etc.
      */
      RtOutputHandler* getOutputHandler() const;

      /**
      \return the output handler managing the outputs, such as point hits cloud, intensities,
              normals, etc.
      */
      virtual ISensorOutputHandler* getOutputHandlerBase() const override;
      
      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual bool addNotification() override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::Lidar );

    protected:
      Lidar();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      LidarModelRef m_model;
  };
}
