/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Component.h>
#include <agxSDK/StepEventListener.h>
#include <agxStream/Serializable.h>

namespace agx
{
  class RigidBody;
}

namespace agxCollide
{
  class Geometry;
}

namespace agxSDK
{
  class Simulation;
  class TerrainToolInstance;
  class Assembly;

  AGX_DECLARE_POINTER_TYPES(TerrainToolInstance);
  AGX_DECLARE_VECTOR_TYPES(TerrainToolInstance);
  typedef agx::SetVector<agxSDK::TerrainToolInstanceRef> TerrainToolInstanceRefSetVector;

  /**
  An agxTerrain::TerrainToolInstance in a simulation, receiving step
  and contact callbacks in a controllable way to avoid interference
  with user step and contact callbacks.
  */
  class AGXPHYSICS_EXPORT TerrainToolInstance : public agx::Component, public agxStream::Serializable
  {
  public:
    /**
    \return simulation this instance is part of, nullptr if not added to one.
    */
    agxSDK::Simulation* getSimulation() const;


    /**
    Add a StepEventListener which follows the shovel - ie added and removed from a simulation at the same time as the
    shovel is added/removed from the simulation.
    */
    void addEventListener(agxSDK::StepEventListener* listener);

    /**
    Remove an added StepEventListener from the shovel. May still be in the active simulation.
    */
    bool removeEventListener(agxSDK::StepEventListener* listener);

    /*
    Check whether the given shovel is valid.
    */
    virtual bool isValid() const = 0;

  public:
    DOXYGEN_START_INTERNAL_BLOCK()

    virtual void onAssemblyAdd( agxSDK::Assembly* assembly ) = 0;

    AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE(agxSDK::TerrainToolInstance);
    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    /**
    Default constructor.
    */
    TerrainToolInstance();

    /**
    Reference counted object - protected destructor.
    */
    virtual ~TerrainToolInstance();


  protected:
    /**
    \internal

    Callback when this instance is added to a simulation.
    */
    virtual void addNotification();

    /**
    \internal

    Callback when this instance is removed from a simulation.
    */
    virtual void removeNotification();

    /**
    \internal

    Pre collide callback.
    */
    virtual void preCollide();

    /**
    \internal

    Pre solver callback.
    */
    virtual void pre();

    /**
    \internal

    Post solver callback.
    */
    virtual void post();

    /**
    \internal

    Callback to be executed at the end of the time step
    */
    virtual void last();

  private:
    /**
    \internal

    Called from simulation when this instance is added or removed.
    */
    void setSimulation(agxSDK::Simulation* simulation);

  private:
    friend class Simulation;
    Simulation* m_simulation;
    friend class TerrainManager;

    agxSDK::StepEventListenerRefVector m_listeners;
  };

}
