/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <utility>
#include <memory>
#include <agxSDK/Assembly.h>
#include <openplx/Object.h>
#include <openplx/OpenPlxCoreApi.h>
#include <agxOpenPLX/export.h>
#include <agxOpenPLX/OpenPlxToAgxMapper.h>
#include <agxOpenPLX/AgxCache.h>
#include <agxOpenPLX/SignalQueue.h>

namespace agxopenplx {

    class OpenPlxToAgxVisualsMapper;

    /**
     * \brief The return type of functions that load a .openplx file. Bundles
     * all the resulting data of loading a .openplx scene.
     */
    class AGXOPENPLX_EXPORT LoadResult {
        public:
            LoadResult() = default;

            LoadResult(agxSDK::Assembly* assembly,
                std::shared_ptr<openplx::Core::Api::OpenPlxContext> context,
                std::shared_ptr<openplx::Core::Object> scene,
                openplx::Errors errors,
                std::shared_ptr<agxopenplx::OpenPlxToAgxMapper> to_agx_mapper,
                std::shared_ptr<agxopenplx::OpenPlxToAgxVisualsMapper> to_agx_visuals_mapper,
                bool visuals_added
            );

            /**
             * \return The assembly into which the openplx scene has been loaded
             */
            agxSDK::Assembly* assembly();

            /**
             * \return The power line into which the openplx drive train has been loaded
             */
            agxPowerLine::PowerLine* power_line();

            /**
             * \return The openplx root instance (usually a Physics3D.System)
             */
            std::shared_ptr<openplx::Core::Object> scene();

            /**
             * \return The openplx errors that was found when parsing/loading the .openplx file
             */
            openplx::Errors errors();

            /**
             * \return The openplx core context
             */
            std::shared_ptr<openplx::Core::Api::OpenPlxContext> context();

            /**
             * \return The input signal queue if signalling is configured, null otherwise
             */
            std::shared_ptr<agxopenplx::InputSignalQueue> getInputSignalQueue();

            /**
             * \return The output signal queue if signalling is configured, null otherwise
             */
            std::shared_ptr<agxopenplx::OutputSignalQueue> getOutputSignalQueue();

            /**
             * \brief True if the loaded model contained any visuals
             */
            bool visualsAdded();

DOXYGEN_START_INTERNAL_BLOCK()
            /**
             * \return The OpenPlxToAgxMapper instance (used internally)
             */
            std::shared_ptr<agxopenplx::OpenPlxToAgxMapper> toAgxMapper();

            /**
             * \return The OpenPlxToAgxVisualsMapper instance (used internally)
             */
            std::shared_ptr<agxopenplx::OpenPlxToAgxVisualsMapper> toAgxVisualsMapper();

            /**
             * \brief Adds a input signal queue to the result (used internally)
             */
            void addInputSignalQueue(std::shared_ptr<agxopenplx::InputSignalQueue> input_queue);

            /**
             * \brief Adds a output signal queue to the result (used internally)
             */
            void addOutputSignalQueue(std::shared_ptr<agxopenplx::OutputSignalQueue> output_queue);
DOXYGEN_END_INTERNAL_BLOCK()

        private:
            agxSDK::AssemblyRef m_assembly;
            std::shared_ptr<openplx::Core::Api::OpenPlxContext> m_context;
            std::shared_ptr<openplx::Core::Object> m_scene;
            openplx::Errors m_errors;
            std::shared_ptr<agxopenplx::OpenPlxToAgxMapper> m_to_agx_mapper;
            std::shared_ptr<agxopenplx::OpenPlxToAgxVisualsMapper> m_to_agx_visuals_mapper;
            std::shared_ptr<agxopenplx::InputSignalQueue> m_input_queue;
            std::shared_ptr<agxopenplx::OutputSignalQueue> m_output_queue;
            bool m_visuals_added;
    };

    /**
     * model_name - if set a specific model from the source/file will be loaded instead of the default
     * uuidv5_namespace_id - if set objects will be assigned version 5 UUIDs based on the namespace id
     *     and the object name instead of the default version 4 UUID.
     * map_visuals - if set, visuals are included in the mapping.
     */
    class AGXOPENPLX_EXPORT OptParams {
        public:
            OptParams with_model_name(const std::string& model_name);
            OptParams with_uuidv5(const std::string& uuidv5_namespace_id);
            OptParams with_map_visuals(bool map_visuals = true);
            OptParams with_debug_render_frames(bool debug_render_frames = true);

            const std::optional<std::string>& model_name() const;
            const std::optional<std::string>& uuidv5_namespace_id() const;
            bool map_visuals() const;
            bool debug_render_frames() const;
        private:
            std::optional<std::string> m_model_name = std::nullopt;
            std::optional<std::string> m_uuidv5_namespace_id = std::nullopt;
            bool m_map_visuals {false};
            bool m_debug_render_frames {false};
    };

    /**
     * @brief Loads scene from file and report errors if any
     *
     * Registers bundles, parses OpenPLX and maps OpenPLX to agx
     *
     * @param simulation The agx simulation to load the scene into
     * @param source The .openplx source text
     * @param bundle_paths A ; or : separated list of paths to the bundles.
     * @param optional_parameters Optional parameters, see agxopenplx::OptParams
     */
    AGXOPENPLX_EXPORT LoadResult load_from_string(agxSDK::Simulation *simulation, const std::string& source, const std::string& bundle_paths,
                             OptParams optional_parameters = OptParams());

     /**
     * @brief Loads scene from string and report errors if any
     * Registers bundles, parses OpenPLX and maps OpenPLX to agx
     *
     * @param simulation The agx simulation to load the scene into
     * @param path The path to the .openplx file
     * @param bundle_paths A ; or : separated list of paths to the bundles.
     * @param optional_parameters Optional parameters, see agxopenplx::OptParams
     */
    AGXOPENPLX_EXPORT LoadResult load_from_file(agxSDK::Simulation *simulation, const std::string& path, const std::string& bundle_paths,
                           OptParams optional_parameters = OptParams());

    /**
     * Serializes the runtime tree to JSON
     */
    AGXOPENPLX_EXPORT std::string serialize_file(const std::string& path, const std::string& bundle_paths,
                                     OptParams optional_parameters = OptParams());

    /**
     * @brief Register all known OpenPLX bundles
     */
    AGXOPENPLX_EXPORT void register_bundles(openplx::Core::Api::OpenPlxContext& ctx);

    /**
     * @brief Register all known OpenPLX plugins
     */
    AGXOPENPLX_EXPORT void register_plugins(openplx::Core::Api::OpenPlxContext& ctx, std::shared_ptr<agxopenplx::AgxCache> cache = nullptr);

    /**
     * @brief Set openplx log level
     *
     * @param log_level_name One of "trace", "debug", "info", "warn", "error", "critical", or "off".
     */
    AGXOPENPLX_EXPORT void set_log_level(const std::string& log_level_name);

    /**
     * Checks the errors if there are dependency errors that might imply that the user should
     * run migrate.
     * @param current_version The expected version of bundles
     * @param errors The errors reported after parsing/loading
     * @return a non-empty string with the old version if a hint is justified, empty string otherwise
     */
    AGXOPENPLX_EXPORT std::string check_if_migrate_hint_is_justified(const std::string& current_version, const openplx::Errors& errors);

    AGXOPENPLX_EXPORT std::pair<std::shared_ptr<openplx::Core::Object>, openplx::Errors>
    parse_and_evaluate_string(const std::string& bundle_paths, const std::string& source, const std::string& modelname);

    AGXOPENPLX_EXPORT std::string parseWithPlugin(std::shared_ptr<openplx::Plugin>, const std::string& key);

    /**
     * Converts errors to a vector of strings
     * @return a vector of error messages
     */
    AGXOPENPLX_EXPORT std::vector<std::string> get_error_strings(const openplx::Errors& errors);
}
