#pragma once

#include <memory>
#include <vector>
#include <cstdint>
#include <openplx/control_export.h>

namespace openplx {

    namespace Physics::Signals {
        class Input;
        class Output;
    }

    class Marshalling;
    class ControlInterface;
    class HeapControlInterfaceImpl;

    struct HeapControlInterfaceReadBuffer {
        uint8_t* buffer_p;
        size_t size;
    };

    class OPENPLX_CONTROL_EXPORT HeapControlInterface {
        public:
            HeapControlInterface(std::shared_ptr<ControlInterface> control_interface);
            ~HeapControlInterface();

            /**
             * Returns a marshalling instance that can be used to read the data from the output. Previous
             * marshalling instances become invalid after a subsequent call to read.
             */
            std::shared_ptr<Marshalling> read(const std::shared_ptr<Physics::Signals::Output>& output);

            /**
             * Returns a marshalling instance that can be used to read the data from the output. Previous
             * marshalling instances become invalid after a subsequent call to read.
             */
            std::shared_ptr<Marshalling> read_from_key(uint32_t key);

            /**
             * Reads data from all output controls and populate the read buffer.
             * Returns the pointer to the underlying data for the read buffer.
             * The returned pointer is only valid while the HeapControlInterface is
             * active and until the next call to read or read_all.
             * @param keys The set of keys to read, or empty to read from all output controls
             * @return A HeapControlInterface pointing to the internal read buffer.
             */
            HeapControlInterfaceReadBuffer read_multiple(const std::vector<uint32_t>& keys);

            /**
             * Read multiple output controls and return the raw read buffer. Reuses
             * the prepared read buffer from a previous call to read_multiple.
             *
             * Note: This only works if read_multiple has been called
             * at least once and that all output controls expects to write the same
             * number of bytes.
             * @return A HeapControlInterface pointing to the internal read buffer.
             */
            HeapControlInterfaceReadBuffer read_multiple_reuse_buffer();

            /**
             * Returns a marshalling object that can be used to write data that will be used to update
             * the runtime.
             */
            std::shared_ptr<Marshalling> prepare_write(const std::shared_ptr<Physics::Signals::Input>& input);

            /**
             * Commits the writes made to inputs to the runtime. All marshalling instances previously returned from
             * prepare_write becomes invalid.
             */
            void flush();

            /**
             * @return the pointer to the wrapped ControlInterface
             */
            const std::shared_ptr<ControlInterface>& get_control_interface() const;

        private:
            HeapControlInterfaceImpl* m_impl;
    };
}
