/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/Terrain.h>
#include <agx/Journal.h>
#include <agxSDK/StepEventListener.h>

namespace agxTerrain
{

  /**
  Utility class used to record agxTerrain::Terrain height data changes to an agx::Journal.
  This object will be serialized and restore the height values in the coupled Terrain object
  in journal playback. Simply add the class to the simulation before enabling recording:

    agxTerrain::TerrainJournalRecorderRef recorder = new agxTerrain::TerrainJournalRecorder(terrain)
    sim->add(recorder)
  */
  AGX_DECLARE_POINTER_TYPES( TerrainJournalRecorder );
  class AGXTERRAIN_EXPORT TerrainJournalRecorder : public agxSDK::StepEventListener
  {
  public:
    /**
    Finds TerrainJournalRecorder given terrain and simulation.
    \return TerrainJournalRecorder if exist, otherwise nullptr
    */
    static TerrainJournalRecorder* find( const agxTerrain::Terrain* terrain, const agxSDK::Simulation* simulation );

    /**
    Find all active terrains in the simulation and add a TerrainJournalRecorder objects for each one.
    \param simulation The simulation to retrieve the terrains from
    \param useLiveUpdates If true, then will store changes vertices even without a journal, and uses "heights" instead of uuid as buffer name
                          NOTE: This implementation only supports one terrain!
    */
    static size_t addTerrainJournalRecorders( agxSDK::Simulation* simulation, bool useLiveUpdates = false );

    /**
    Constructor for the TerrainJournalRecorder.
    \param terrain The terrain to store in the journal
    \param useLiveUpdates If true, then will store changes vertices even without a journal, and uses "heights" instead of uuid as buffer name
                          NOTE: This implementation only supports one terrain!
    */
    TerrainJournalRecorder( agxTerrain::Terrain* terrain, bool useLiveUpdates = false );

    /**
    Called before the collision detection step is taken in the simulation
    \param time - the current simulation time
    */
    virtual void preCollide( const agx::TimeStamp& time ) override;

    /**
    Called before a step is taken in the simulation
    \param time - the current simulation time
    */
    virtual void pre( const agx::TimeStamp& time ) override;

    /**
    Called after a step is taken in the simulation
    \param time - the current simulation time
    */
    virtual void post( const agx::TimeStamp& time ) override;

    /// Called when this object is added to a simulation
    virtual void addNotification() override;

    /// Get the coupled terrain object
    agxTerrain::Terrain* getTerrain();

    AGXSTREAM_DECLARE_SERIALIZABLE( agxTerrain::TerrainJournalRecorder );

  protected:

    bool heigthBufferIsValid();

    void storeHeights();

    void storeChangedVertices();

    void restoreHeights();

    void initHeightFieldStorage();

    void createJournalBindings( agx::Journal* journal );

    void journalAttachedCallback( agxSDK::Simulation* simulation, agx::Journal* journal );

    agxData::Buffer* getHeightBuffer();

    virtual ~TerrainJournalRecorder();

  private:
    TerrainJournalRecorder();

    //////////////////////////////////////////////////////////////////////////
    // Variables
    //////////////////////////////////////////////////////////////////////////
  protected:
    agxSDK::Simulation::JournalAttachEvent::CallbackType m_journalAttachedCallback;
    agxTerrain::TerrainRef m_terrain;
    agx::RealVector        m_heightIndices;
    agxData::BufferRef     m_heigthBuffer;
    agx::ComponentRef      m_dataRoot;
    agx::Uuid              m_dataRootUUID;
    agx::JournalObserver   m_journal;
    bool                   m_useLiveUpdates;
  };
}
