/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceEntityId.h>
#include <agxSensor/RaytraceComponents.h>

#include <agx/Singleton.h>
#include <agx/HashFunction.h>

#include <agxStream/OutputArchive.h>
#include <agxStream/InputArchive.h>

#include <deque>
#include <memory>

namespace agxSensor
{
  /**
  Storage vector mapping entity ID:s to corresponding values of the specified template type.
  */
  template<typename T>
  class RtStorage : public std::vector<T>
  {
    public:
      using std::vector<T>::vector;

      using std::vector<T>::operator[];

      /**
      Fetches the value for at the supplied index.
      \param index - index of the value to fetch
      \return stored value at the supplied index location or the default value of the storage type
              if the supplied index is not within the stored range
      */
      template<typename IndexT>
      const T& get( IndexT index ) const;

      /**
      Fetches the value for at the supplied entity ID.
      \param entity - ID of the entity whose stored value to fetch
      \return stored value for the supplied entity ID or the default value of the storage type if 
              the supplied entity ID does not have a stored value.
      */
      const T& get( RtEntityId entity ) const;

      /**
      Directly references an existing value entry for the supplied entity ID.
      \param entity - ID of the entity whose stored value entry to reference
      \return reference to the entry
      */
      T& operator[] ( RtEntityId entity )
      {
        agxAssert( (size_t)indexOf( entity ) < this->size() );
        return (*this)[ indexOf( entity ) ];
      }

      /**
      Directly references an existing value entry for the supplied entity ID.
      \param entity - ID of the entity whose stored value entry to reference
      \return reference to the entry
      */
      const T& operator[] ( RtEntityId entity ) const
      {
        agxAssert( (size_t)indexOf( entity ) < this->size() );
        return (*this)[ indexOf( entity ) ];
      }
  };

  template<typename T>
  template<typename IndexT>
  const T& RtStorage<T>::get( IndexT index ) const
  {
    static const T defaultValue = T{ T::DefaultValue };
    return (size_t)index < this->size() ? (*this)[ (size_t)index ] : defaultValue;
  }

  template<typename T>
  inline const T& RtStorage<T>::get( RtEntityId entity ) const
  {
    return this->get( indexOf( entity ) );
  }

  DOXYGEN_START_INTERNAL_BLOCK()
  struct AGXSENSOR_EXPORT RtStorageData
  {
    using StoragePtr = std::shared_ptr<void>;
    using AppendFunc = std::function<void()>;
    using ResetFunc = std::function<void( RtEntityId )>;
    using SizeFunc = std::function<size_t()>;
    using StoreFunc = std::function<void( agxStream::OutputArchive& )>;
    using RestoreFunc = std::function<void( agxStream::InputArchive& )>;

    StoragePtr instance;
    const char* elementName;
    size_t elementSize;
    AppendFunc appendDefault;
    ResetFunc resetElementToDefault;
    SizeFunc size;
    StoreFunc store;
    RestoreFunc restore;

    template<typename T>
    RtStorage<T>& storage()
    {
      return *static_cast<RtStorage<T>*>( this->instance.get() );
    }
  };
  DOXYGEN_END_INTERNAL_BLOCK()

  /**
  Registry allowing value storage mapped to raytrace entities.
  \warning Storing and restoring an agxSensor::RtRegistry may only be conducted using the same 
  standard library implementation version! Stored agxSensor::RtRegistry containers are thus not 
  necessarily transferrable from one system to another!
  */
  class AGXSENSOR_EXPORT RtRegistry : public agx::Singleton
  {
    public:
      using EntityIdVector = std::vector<RtEntityId>;
      using EntityIdQueue = std::deque<RtEntityId>;
      using EntityIdSet = agx::HashSet<RtEntityId>;

    public:
      /**
      Creates a new entity ID to use for mapping.
      \return newly created entity ID
      */
      static RtEntityId createId();

      /**
      Un-registers the supplied entity ID.
      \param entityId - ID to un-register
      */
      static void destroyId( RtEntityId entityId );

      /**
      \return true if the specified entity ID is un-registered from value mapping or false if the
      ID is currently in use, or has never been used for value mapping.
      */
      static bool isDestroyed( RtEntityId entityId );

      /**
      Fetches or creates a new storage for the specified type.
      */
      template<typename T>
      static RtStorage<T>& getOrCreateStorage();

      /**
      Fetches, but does not create, the storage for the specified type.
      */
      template<typename T>
      static const RtStorage<T>& storage();

      DOXYGEN_START_INTERNAL_BLOCK()

      static void store( agxStream::OutputArchive& out );
      static void restore( agxStream::InputArchive& in );

      static void cleanup();

    public:
      SINGLETON_CLASSNAME_METHOD();

      virtual void shutdown() override;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      using StorageTable = agx::HashVector<size_t, RtStorageData>;

    private:
      static RtRegistry& self();

    private:
      RtRegistry();

      void initialize();

      RtEntityId create();
      void destroy( RtEntityId entityId );

      template<typename T>
      RtStorage<T>& assureStorage();

    private:
      EntityIdVector m_entityIds;
      EntityIdQueue m_freeEntityIds;
      EntityIdSet m_destroyedEntityIds;
      StorageTable m_storages;
  };

  template<typename T>
  RtStorage<T>& RtRegistry::assureStorage()
  {
    static constexpr auto& type = typeid( T );
    static constexpr auto defaultValue = T::DefaultValue;

    auto it = m_storages.find( type.hash_code() );
    if ( it == m_storages.end() ) {
      it = m_storages.insert<RtStorageData>(
                              type.hash_code(),
                              {
                                std::make_shared<RtStorage<T>>(),
                                type.name(),
                                sizeof( T ),
                                {},
                                {},
                                {},
                                {},
                                {}
                              } );
      auto storage = &( it->second.storage<T>() );

      it->second.appendDefault = [storage]()
      {
        storage->push_back( T{ defaultValue } );
      };
      it->second.resetElementToDefault = [storage]( RtEntityId entity )
      {
        (*storage)[ entity ] = T{ defaultValue };
      };
      it->second.size = [storage]()
      {
        return storage->size();
      };
      it->second.store = [storage]( agxStream::OutputArchive& out )
      {
        out << agxStream::out( "numElements", storage->size() );
        out.beginSection( "storageData" );
        out.write( storage->data(), sizeof( T ) * storage->size() );
        out.endSection( "storageData" );
      };
      it->second.restore = [storage]( agxStream::InputArchive& in )
      {
        size_t numElements = 0u;
        in >> agxStream::in( "numElements", numElements );
        storage->resize( numElements );
        in.beginSection( "storageData" );
        in.read( storage->data(), sizeof( T ) * numElements );
        in.endSection( "storageData" );
      };

      while ( storage->size() < m_entityIds.size() )
        it->second.appendDefault();
    }

    return it->second.storage<T>();
  }

  template<typename T>
  RtStorage<T>& RtRegistry::getOrCreateStorage()
  {
    return self().template assureStorage<T>();
  }

  template<typename T>
  const RtStorage<T>& RtRegistry::storage()
  {
    static constexpr auto& type = typeid( T );
    static const auto emptyStorage = RtStorage<T>{};

    auto it = self().m_storages.find( type.hash_code() );

    if ( it == self().m_storages.end() )
      return emptyStorage;

    return it->second.storage<T>();
  }
}
