/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/TriaxialSignalSystemNode.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( GyroscopeLinearAccelerationEffects );
  AGX_DECLARE_POINTER_TYPES( GyroscopeLinearAccelerationEffectsNode );

  /**
  Applies an offset to the zero rate bias depending on the linear acceleration that the gyroscope is
  exposed to.
  */
  class AGXSENSOR_EXPORT GyroscopeLinearAccelerationEffects : public ITriaxialSignalSystemNode
  {
    public:
      /**
      Constructs this gyroscope linear acceleration effects by specifying the effect tangent to each
      of the measurement axes.
      \param accelerationEffects - linear acceleration effects along each of the measurement axes
      */
      GyroscopeLinearAccelerationEffects( const agx::Vec3& accelerationEffects =
                                            agx::Vec3( 2.97 * 1e-4 ) );

      /**
      Constructs this gyroscope linear acceleration effects node using the given
      \p accelerationEffects effect matrix.
      \param accelerationEffects - matrix describing the linear acceleration effects from each axis
                                   onto every other axis
      */
      GyroscopeLinearAccelerationEffects( const agx::Matrix3x3& accelerationEffects );

      /**
      Sets the linear acceleration effects tangent to each of the measurement axes.
      \param accelerationEffects - linear acceleration effects along each of the measurement axes
      */
      void setAccelerationEffects( const agx::Vec3& accelerationEffects );

      /**
      Specifies the entire linear acceleration effects matrix.
      \param accelerationEffects - matrix describing the linear acceleration effects from each axis
                                   onto every other axis
      */
      void setAccelerationEffects( const agx::Matrix3x3& accelerationEffects );

      /**
      \return linear acceleration effects matrix
      */
      const agx::Matrix3x3& getAccelerationEffectsMatrix() const;

      /**
      \return new, unique, triaxial signal system node (proxy) of this implementation
      */
      virtual TriaxialSignalSystemNodeRef createTriaxialNodeProxy() override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( SystemNodeProxy& proxy, const CallbackData& data ) override;
      virtual void cleanup( SystemNodeProxy& proxy ) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::GyroscopeLinearAccelerationEffects );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Matrix3x3 m_effectsMatrix;
  };

  /**
  Instance proxy node for storing the state information need by the gyroscope linear acceleration
  effects.
  */
  class AGXSENSOR_EXPORT GyroscopeLinearAccelerationEffectsNode : public TriaxialSignalSystemNode
  {
    public:
      friend GyroscopeLinearAccelerationEffects;

    public:
      /**
      Construct a triaxial signal system node instance with the given \p implementation.
      \param implementation - behavioral implementation for this proxy (should not be nullptr)
      */
      GyroscopeLinearAccelerationEffectsNode( GyroscopeLinearAccelerationEffects* implementation );

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      GyroscopeLinearAccelerationEffectsNode();

      virtual void synchronize( const CallbackData& data ) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::GyroscopeLinearAccelerationEffectsNode );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Vec3 m_preIntegrationVelocity;
      agx::Vec3 m_preIntegrationAngularVelocity;
  };
}
