/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Referenced.h>

#include <agxStream/Serializable.h>

namespace agxSDK
{
  class Simulation;

  /**
  Interface to a cooperative sensor simulation assumed to be executed
  in a separate thread spawned in addNotification and joined in
  removeNotification.

  The sensor simulation step is called after the user step callback
  preCollide and spans until user step post (over collision detection,
  pre callbacks and solve dynamics) when fetchStepResults is called.

  I.e., block read data from main resources (e.g., agxSDK::Simulation)
  in 'step' and activate the simulation thread. Synchronize with main
  in 'fetchStepResults' so the users can start collecting data in post,
  last events or outside the simulation loop.
  */
  class AGXPHYSICS_EXPORT ISensorEnvironment : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Default constructor.
      */
      ISensorEnvironment() = default;

      /**
      Destructor.
      */
      virtual ~ISensorEnvironment() = default;

      /**
      \return the simulation this sensor environment belongs to,
              is nullptr in standalone mode
      */
      const Simulation* getSimulation() const;

    public:
      /**
      Called when added to a simulation or explicitly called when
      the sensor simulation should start. This call should create
      the simulation thread and sleep until 'step' is called.
      */
      virtual void addNotification() = 0;

      /**
      Called when removed from a simulation or explicitly called when
      the sensor simulation should stop/join the main thread.
      */
      virtual void removeNotification() = 0;

      /**
      Start the sensor simulation and step for delta time \p dt. It's
      safe to read data from the simulation or other reasources synchronized
      with the main thread.
      \param dt - delta time to step the sensor simulation
      */
      virtual void step( agx::Real dt ) = 0;

      /**
      Block main thread until the sensor simulation is done and the
      data/results/outputs are accessible.
      */
      virtual void fetchStepResults() = 0;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSDK::ISensorEnvironment );

    private:
      ISensorEnvironment( const ISensorEnvironment& ) = delete;
      ISensorEnvironment( ISensorEnvironment&& ) = delete;
      ISensorEnvironment& operator= ( const ISensorEnvironment& ) = delete;

    private:
      friend class Simulation;

      void setSimulation( Simulation* simulation );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::observer_ptr<agx::Referenced> m_simulation;
  };
}
