/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxOpenPLX/export.h>

#include <agx/Referenced.h>

#include <memory>
#include <optional>
#include <unordered_map>
#include <variant>

namespace agxopenplx
{
  /**
   * @brief Base for non-AGX Dynamics dynamic types stored as metadata.
   */
  class AgxMetadataDynamic
  {
    public:
      virtual ~AgxMetadataDynamic() noexcept = default;
  };

  /**
   * @brief Storage of metadata tied to AGX Dynamics objects as needed by the agxOpenPLX runtime.
   */
  class AGXOPENPLX_EXPORT AgxMetadata
  {
    public:
      enum MetadataType : std::size_t
      {
        // AGX Dynamics referenced
        REFERENCED,
        // C++ dynamic type
        DYNAMIC,
        // Raw memory
        MEMORY
      };

      using Metadata =
        std::variant<agx::ref_ptr<agx::Referenced>, std::shared_ptr<AgxMetadataDynamic>, std::shared_ptr<char[]>>;

    public:
      /**
       * @brief Construct an empty metadata storage.
       */
      AgxMetadata() = default;

      // YES!Move
      AgxMetadata(AgxMetadata&&) noexcept = default;
      AgxMetadata& operator=(AgxMetadata&&) noexcept = default;

      // NO!Copy
      AgxMetadata(const AgxMetadata&) = delete;
      AgxMetadata& operator=(const AgxMetadata&) noexcept = delete;

    public:
#ifndef SWIG
      /**
       * @brief Fetch metadata associated with the given AGX object.
       * @param agx_object - AGX Dynamics object with stored metadata.
       * @return the associated metadata or nullopt if no metadata has been mapped.
       */
      std::optional<Metadata> getMetadata(agx::Referenced* agx_object);

      /**
       * @brief Fetch metadata of the specified type associated with the given AGX object.
       * @param agx_object - AGX Dynamics object with stored metadata.
       * @return the associated metadata or nullopt if no metadata of the specific type has been mapped.
       */
      template <MetadataType Type>
      std::optional<std::variant_alternative_t<Type, Metadata>> getMetadata(agx::Referenced* agx_object);

      /**
       * @brief Register another AGX Dynamics object as the metadata connected to the given AGX Dynamics object.
       * @param agx_object - AGX Dynamics object which to associate the metadata with.
       * @param metadata - Metadata to associate to the given AGX Dynamics object.
       */
      void registerMetadata(agx::Referenced* agx_object, agx::ref_ptr<agx::Referenced> metadata);

      /**
       * @brief Register a non-AGX Dynamics dynamic type as the metadata connected to the given AGX Dynamics object.
       * @param agx_object - AGX Dynamics object which to associate the metadata with.
       * @param metadata - Metadata to associate to the given AGX Dynamics object.
       */
      void registerMetadata(agx::Referenced* agx_object, std::shared_ptr<AgxMetadataDynamic> metadata);

      /**
       * @brief Register a raw memory block as the metadata connected to the given AGX Dynamics object.
       * @param agx_object - AGX Dynamics object which to associate the metadata with.
       * @param metadata - Metadata to associate to the given AGX Dynamics object.
       */
      void registerMetadata(agx::Referenced* agx_object, std::shared_ptr<char[]> metadata);

#endif

    private:
      struct MetadataEntry
      {
          agx::observer_ptr<agx::Referenced> object;
          Metadata metadata;
      };

    private:
      std::unordered_map<agx::Referenced*, MetadataEntry> m_storage;
  };

#ifndef SWIG
  template <AgxMetadata::MetadataType Type>
  inline std::optional<std::variant_alternative_t<Type, AgxMetadata::Metadata>> AgxMetadata::getMetadata(
    agx::Referenced* agx_object)
  {
    std::optional<AgxMetadata::Metadata> metadata = getMetadata(agx_object);
    if (metadata && metadata->index() == Type)
      return std::get<Type>(*metadata);
    else
      return std::nullopt;
  }
#endif
}
