/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxOSG/export.h>

#include <agx/PushDisableWarnings.h> // Disabling warnings. Include agx/PopDisableWarnings.h below!
#include <osg/MatrixTransform>
#include <osg/Geode>
#include <agx/Vector.h>
#include <osg/TextureCubeMap>
#include <agx/PopDisableWarnings.h> // End of disabled warnings.

namespace agxOSG
{
  /**
  Create a Sphere with a specified name that will project a cubemap spherically (not as 6 faces) using a shader. 
  The sphere will always follow the camera.
  \param name - Name of the node
  \param images - 6 images each with a 90x90 fov view of the environment.  Order: +x, -x, +y, -y, +z. -z
  \param radius - Radius of the sphere
  \param offset - Offset in up/down where 0 mean the camera will be at the center of the sphere. Negative will move the sphere down relative the camera.
  \return pointer to the created node.
  */
  AGXOSG_EXPORT osg::Geode*
    createSkySphere( const std::string& name,  agx::Vector<osg::ref_ptr<osg::Image> >& images, float radius, float offset=0 );

  /**
  Create a TextureCubeMap from 6 images with a 90x90 fov
  \param images - 6 images each with a 90x90 fov view of the environment. Order: +x, -x, +y, -y, +z. -z
  \return pointer to the cubemap
  */
  AGXOSG_EXPORT osg::TextureCubeMap *createTextureCubeMap( agx::Vector<osg::ref_ptr<osg::Image> >& images);

  /**
  Creates an returns a CubeMap based on the images.
  \param imagename - Prefix name of the 6 textures for the cubemap.Filenames will be <imagename>+_posx.<filetype>, _negx, posy, negy, ...
  \param filetype - Filetype of the images ".dds", ".png"
  \return pointer to a TextureCubeMap pointer
  */
  AGXOSG_EXPORT osg::TextureCubeMap *createTextureCubeMap( const std::string& imagename, const std::string& filetype );

  /**
  Creates a skybox. 
  
  Note: Only works in ShaderMode.

  \param imagename - Prefix name of the 6 textures for the cubemap. Filenames will be <image_name>+_posx.<filetype>, _negx, posy, negy, ...
  \param filetype - Filetype of the images ".dds", ".png"
  \param size - Size of Skybox
  \param offset - Offset in z. For example if you want the "floor" of the skybox to align with a plane geometry in your scene. This will move your skybox up and down.
  */
  class AGXOSG_EXPORT SkyBox : public osg::Group
  {
    public:
      /// Constructor
      SkyBox(const std::string& name,
        const std::string& imagename,
        const std::string& filetype,
        double size = 1000,
        float offset = 0
      );

    protected:
      double m_size;
  };
}
