/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/agx.h>

#ifdef _MSC_VER
#else
#ifdef __APPLE__
#include <CoreFoundation/CoreFoundation.h>
#else //LINUX
// The byteswap_.* family of macros inject code containing the deprecated 'register' keyword.
#include <agx/PushDisableWarnings.h>
#include <byteswap.h>
#endif
#endif

namespace agx
{

  /**
  Swap all the bytes in an array to convert from little - endian
  byte order to big-endian byte order, or vice versa.
  \note           Works for arrays of any size. Swaps the bytes **in place** in the array.
  \param[in,out]  byte_array  The array in which to swap the bytes in-place.
  \param[in]      len         The length (in bytes) of the array.
  \return         None
  */
  static void ByteSwapArray( uint8_t* byte_array, size_t len )
  {
    size_t i_left = 0;  // index for left side of the array
    size_t i_right = len - 1;  // index for right side of the array
    while (i_left < i_right)
    {
      // swap left and right bytes
      uint8_t left_copy = byte_array[i_left];
      byte_array[i_left] = byte_array[i_right];
      byte_array[i_right] = left_copy;
      i_left++;
      i_right--;
    }
  }

  static inline double ByteSwap64( const double& x )
  {
    double val = x;
    ByteSwapArray((uint8_t*)(&val), sizeof(val));
    return val;
  }

  static inline float ByteSwap32( const float& x )
  {
    float val = x;
    ByteSwapArray((uint8_t*)(&val), sizeof(val));
    return val;
  }

#ifdef _MSC_VER
  static inline uint16_t ByteSwap16( const uint16_t& x )
  {
    uint16_t result = x;
    ByteSwapArray((uint8_t*)&result, sizeof(result));
    return result;
  }

  static inline int16_t ByteSwap16( const int16_t& x )
  {
    int16_t result = x;
    ByteSwapArray((uint8_t*)&result, sizeof(result));
    return result;
  }

  static inline uint32_t ByteSwap32( const uint32_t& x )
  {
    uint32_t result = x;
    ByteSwapArray((uint8_t*) & result, sizeof(result));
    return result;
  }

  static inline int32_t ByteSwap32(const int32_t& x)
  {
    uint32_t result = x;
    ByteSwapArray((uint8_t*)&result, sizeof(result));
    return result;
  }

  static inline uint64_t ByteSwap64 ( const uint64_t& x )
  {
    uint64_t result = x;
    ByteSwapArray((uint8_t*)&result, sizeof(result));
    return result;
  }

  static inline int64_t ByteSwap64(const int64_t& x)
  {
    int64_t result = x;
    ByteSwapArray((uint8_t*)&result, sizeof(result));
    return result;
  }


#else
#ifdef __APPLE__
  static inline uint16_t ByteSwap16( uint16_t x )
  {
    return CFSwapInt16(x);
  }

  static inline int16_t ByteSwap16( int16_t x )
  {
    return (int16_t)CFSwapInt16((uint16_t)x);
  }

  static inline uint32_t ByteSwap32( uint32_t x )
  {
    return CFSwapInt32(x);
  }

  static inline int32_t ByteSwap32( int32_t x )
  {
    return (int32_t)CFSwapInt32((uint32_t)x);
  }

  static inline uint64_t ByteSwap64( uint64_t x )
  {
    return CFSwapInt64(x);
  }

  static inline int64_t ByteSwap64( int64_t x )
  {
    return (int64_t)CFSwapInt64((uint64_t)x);
  }

#else //LINUX

  static inline uint16_t ByteSwap16( uint16_t x )
  {
    return bswap_16(x);
  }

  static inline int16_t ByteSwap16( int16_t x )
  {
    return (int16_t)bswap_16((uint16_t)x);
  }

  static inline uint32_t ByteSwap32( uint32_t x )
  {
    return bswap_32(x);
  }

  static inline int32_t ByteSwap32( int32_t x )
  {
    return (int32_t)bswap_32((uint32_t)x);
  }

  static inline uint64_t ByteSwap64( uint64_t x )
  {
    return bswap_64(x);
  }

  static inline int64_t ByteSwap64( int64_t x )
  {
    return (int64_t)bswap_64((uint64_t)x);
  }

#endif
#endif
}

#ifdef __linux__
#include <agx/PopDisableWarnings.h>
#endif
