
// We should not import Robotec GPU Lidar headers into the API portion of
// Algoryx GPU Sensors because it exposes a bunch of symbols that are hidden
// within libRobotecGPULidar.a and not accessible from e.g. libagxSensor.so. We
// therefor duplicate a bunch of type definitions.
//
// This duplication should only be done for users of Algoryx GPU Sensors. When
// building the library itself, i.e. agpu/src/api/core.cpp, we need access to
// the actual Robotec GPU Lidar and thus its rgl/api/core.h. So the type
// definitions below should not be included if rgl/api/core.h has been included
// already.
//
// This entire thing is ugly, we need to find a better solution for this.
//
// NOTE: If you are going to include rgl/api/core.h in a translational unit then
// it MUST be included before this file so that RGL_VERSION_MAJOR is available.
#if !defined(RGL_VERSION_MAJOR)

#include <stdint.h>

// Default Entity ID is the largest positive 28-bit integer (OptiX limit).
// It is assigned by default if the user does not specify it.
#define RGL_DEFAULT_ENTITY_ID 268435455

/**
 * Two consecutive 32-bit floats.
 */
typedef struct
{
	float value[2];
} rgl_vec2f;

/**
 * Three consecutive 32-bit floats.
 */
typedef struct
{
	float value[3];
} rgl_vec3f;

/**
 * Opaque handle for a computational graph node in RGL.
 * Represents a single computational step, e.g. introducing gaussian noise, raytracing or downsampling.
 * Nodes form a directed acyclic graph, dictating execution order.
 */
typedef struct Node* rgl_node_t;

/**
 * Row-major matrix with 3 rows and 4 columns of 32-bit floats.
 * Right-handed coordinate system.
 */
typedef struct
{
	float value[3][4];
} rgl_mat3x4f;

/**
 * Three consecutive 32-bit signed integers.
 */
typedef struct
{
	int32_t value[3];
} rgl_vec3i;

/**
 * Radar parameters applied at a given distance range.
 */
typedef struct
{
	/**
	 * The beginning distance range for the parameters.
	 */
	float begin_distance;
	/**
	 * The end range distance for the parameters.
	 */
	float end_distance;
	/**
	 * The maximum distance difference to create a new radar detection (in simulation units).
	 */
	float distance_separation_threshold;
	/**
	 * The maximum radial speed difference to create a new radar detection (in simulation units)
	 */
	float radial_speed_separation_threshold;
	/**
	 * The maximum azimuth difference to create a new radar detection (in radians).
	 */
	float azimuth_separation_threshold;
} rgl_radar_scope_t;

/**
 * Radar object class used in object tracking.
 */
typedef enum : int32_t
{
	RGL_RADAR_CLASS_CAR = 0,
	RGL_RADAR_CLASS_TRUCK,
	RGL_RADAR_CLASS_MOTORCYCLE,
	RGL_RADAR_CLASS_BICYCLE,
	RGL_RADAR_CLASS_PEDESTRIAN,
	RGL_RADAR_CLASS_ANIMAL,
	RGL_RADAR_CLASS_HAZARD,
	RGL_RADAR_CLASS_UNKNOWN,
	RGL_RADAR_CLASS_COUNT
} rgl_radar_object_class_t;

/**
 * Represents on-GPU Mesh that can be referenced by Entities on the Scene.
 * Each Mesh can be referenced by any number of Entities on different Scenes.
 */
typedef struct Mesh* rgl_mesh_t;

/**
 * Opaque handle representing an object visible to lidars.
 * An Entity is always bound to exactly one Scene.
 */
typedef struct Entity* rgl_entity_t;

/**
 * Represents on-GPU texture that can be referenced by Entities on the Scene.
 * Each texture can be referenced by any number of Entities on different Scenes.
 */
typedef struct Texture* rgl_texture_t;


/**
 * Opaque handle representing a Scene - a collection of Entities.
 * Using Scene is optional. NULL can be passed to use an implicit default Scene.
 */
typedef struct Scene* rgl_scene_t;


/**
 * Enumerates available extensions in RGL which can be queried using `rgl_get_extension_info`.
 * For more details, see the chapter on extensions in the README.
 * The order of constants must not be changed.
 */
typedef enum : int32_t
{
	RGL_EXTENSION_PCL = 0,
	RGL_EXTENSION_ROS2 = 1,
	RGL_EXTENSION_UDP = 2,
	RGL_EXTENSION_WEATHER = 3,
	RGL_EXTENSION_COUNT
} rgl_extension_t;

/**
 * Status (error) codes returned by all RGL API functions.
 * Unrecoverable errors require reloading the library (restarting the application).
 */
typedef enum : int32_t
{
	/**
	 * Operation successful.
	 * This is a recoverable error :)
	 */
	RGL_SUCCESS = 0,

	/**
	 * One of the arguments is invalid (e.g., null pointer) or a number in an invalid range.
	 * Get the error string for more details.
	 * This is a recoverable error.
	 */
	RGL_INVALID_ARGUMENT,

	/**
	 * RGL's internal state has been corrupted by a previous unrecoverable error.
	 * Application must be restarted.
	 */
	RGL_INVALID_STATE,

	/**
	 * Indicates that a logging operation (e.g., configuration) was unsuccessful.
	 * This is an unrecoverable error.
	 */
	RGL_LOGGING_ERROR,

	/**
	 * Indicates that provided API object handle is not known by RGL.
	 * This can be caused by using previously destroyed API objects, e.g.
	 * by previous call to rgl_*_destroy(...) or rgl_cleanup()
	 * This is a recoverable error.
	 */
	RGL_INVALID_API_OBJECT,

	/**
	* Indicates that a given file could not be opened.
	* This is a recoverable error.
	*/
	RGL_INVALID_FILE_PATH,

	/**
	* Indicates that a tape operation was not successful.
	* This is a recoverable error.
	*/
	RGL_TAPE_ERROR,

	/**
	* Indicates that a UDP operation (e.g., creating socket, sending packet) was not successful.
	* This is a recoverable error.
	*/
	RGL_UDP_ERROR,

	/**
	* Indicates that a ROS2 native library throws an exception.
	* This is a recoverable error.
	*/
	RGL_ROS2_ERROR,

	/**
	 * Indicates an error in the pipeline, such as adjacency of incompatible nodes.
	 * This is a recoverable error.
	 */
	RGL_INVALID_PIPELINE,

	/**
	 * Indicates a failure during (lazy) initialization.
	 * This is an unrecoverable error.
	 */
	RGL_INITIALIZATION_ERROR,

	/**
	 * Requested functionality still needs to be implemented.
	 * This is a recoverable error.
	 */
	RGL_NOT_IMPLEMENTED = 404,

	/**
	 * An unhandled internal error has occurred.
	 * If you see this error, please file a bug report.
	 * This is an unrecoverable error.
	 */
	RGL_INTERNAL_EXCEPTION = 500,
} rgl_status_t;

/**
 * Available logging verbosity levels.
 */
typedef enum : int32_t
{
	RGL_LOG_LEVEL_ALL = 0,
	RGL_LOG_LEVEL_TRACE = 0,
	RGL_LOG_LEVEL_DEBUG = 1,
	RGL_LOG_LEVEL_INFO = 2,
	RGL_LOG_LEVEL_WARN = 3,
	RGL_LOG_LEVEL_ERROR = 4,
	RGL_LOG_LEVEL_CRITICAL = 5,
	RGL_LOG_LEVEL_OFF = 6,
	RGL_LOG_LEVEL_COUNT = 7
} rgl_log_level_t;



/**
 * Available point attributes used to specify the layout of the binary data.
 */
typedef enum : int32_t
{
	RGL_FIELD_XYZ_VEC3_F32 = 1,
	/**
	 * Strength of the returned signal captured by the LiDAR sensor.
	 * It is simulated using intensity textures assigned to entities (see `rgl_entity_set_intensity_texture`).
	 * The final value also depends on the incident angle of the ray hit.
	 */
	RGL_FIELD_INTENSITY_F32,
	/**
	 * Same as RGL_FIELD_INTENSITY_F32, but uint8_t type.
	 */
	RGL_FIELD_INTENSITY_U8,
	RGL_FIELD_IS_HIT_I32,
	RGL_FIELD_IS_GROUND_I32,
	RGL_FIELD_RAY_IDX_U32,
	RGL_FIELD_ENTITY_ID_I32,
	RGL_FIELD_DISTANCE_F32,
	/**
	 * Azimuth angle of the hit point in radians.
	 * Currently only compatible with engines that generate rays as follows:
	 * uses a left-handed coordinate system, rotation applies in ZXY order, up vector is Y, forward vector is Z
	 */
	RGL_FIELD_AZIMUTH_F32,
	/**
	 * Elevation angle of the hit point in radians.
	 * Currently only compatible with engines that generate rays as follows:
	 * uses a left-handed coordinate system, rotation applies in ZXY order, up vector is Y, forward vector is Z
	 */
	RGL_FIELD_ELEVATION_F32,
	RGL_FIELD_RING_ID_U16,
	RGL_FIELD_RETURN_TYPE_U8,
	/**
	 * Seconds have passed since the time of the sensor trigger when this point was measured.
	 * If velocity distortion is disabled, the time stamp for all points will be zero.
	 */
	RGL_FIELD_TIME_STAMP_F64,
	/**
	 * Nanoseconds have passed since the time of the sensor trigger when this point was measured.
	 * If velocity distortion is disabled, the time stamp for all points will be zero.
	 */
	RGL_FIELD_TIME_STAMP_U32,

	/**
	 * Velocity of the hit point on the entity.
	 * It depends on entity's
	 * - linear velocity
	 * - angular velocity
	 * - mesh deformations (e.g. skinning)
	 * The aforementioned are inferred from calls to `rgl_entity_set_pose`, `rgl_scene_set_time` and `rgl_mesh_update_vertices`.
	 */
	RGL_FIELD_ABSOLUTE_VELOCITY_VEC3_F32,

	/**
	 * Velocity of the hit point on the entity, in the coordinate frame of rays source.
	 */
	RGL_FIELD_RELATIVE_VELOCITY_VEC3_F32,

	/**
	 * Scalar describing distance increase per second between hit point and ray source.
	 */
	RGL_FIELD_RADIAL_SPEED_F32,

	/**
	 * Radar-specific fields. At the moment, placeholders only to implement in the future.
	 */
	RGL_FIELD_POWER_F32,
	RGL_FIELD_RCS_F32, // Radar cross-section
	RGL_FIELD_NOISE_F32,
	RGL_FIELD_SNR_F32, // Signal-to-noise ratio

	/**
	 * Normal vector of the mesh triangle where the hit-point is located.
	 * Assumes right-hand rule of vertices ordering.
	 */
	RGL_FIELD_NORMAL_VEC3_F32,

	/**
	 * Incident angle of the ray hitting the mesh triangle in radians.
	 * In range [0, PI/2) rad, where 0 means the ray hit the triangle perpendicularly.
	 */
	RGL_FIELD_INCIDENT_ANGLE_F32,

	/**
	 * 3x4 matrix describing pose of the ray in the world coordinate system.
	 */
	RGL_FIELD_RAY_POSE_MAT3x4_F32,

	/**
	 * Lidar reflective value. Similar to the `RGL_FIELD_INTENSITY_F32` but set as a single value for the entire entity.
	 * Could be replaced with `RGL_FIELD_INTENSITY_F32` and a 1x1 texture when float-type texture will be supported.
	 * For non-hit points zero is assigned.
	 */
	RGL_FIELD_LASER_RETRO_F32,

	// Dummy fields
	RGL_FIELD_PADDING_8 = 1024,
	RGL_FIELD_PADDING_16,
	RGL_FIELD_PADDING_32,
	// Dynamic fields
	RGL_FIELD_DYNAMIC_FORMAT = 13842,
} rgl_field_t;

/**
 * Kinds of return type for multi-return LiDAR output.
 */
typedef enum : int32_t
{
	RGL_RETURN_TYPE_NOT_DIVERGENT = 0,
	RGL_RETURN_TYPE_FIRST = 1,
	RGL_RETURN_TYPE_LAST = 2,
} rgl_return_type_t;

/**
 * Helper enum for axis selection
 */
typedef enum : int32_t
{
	RGL_AXIS_X = 1,
	RGL_AXIS_Y = 2,
	RGL_AXIS_Z = 3,
} rgl_axis_t;

// End of fallback definitions of Robotec GPU Lidar types.
#endif
