
#pragma once

#include <vector>
#include <filesystem>
#include <optional>
#include <openplx/BundleConfig.h>
#include <openplx/BundleLookup.h>
#include <openplx/EvaluatorContext.h>

namespace openplx::Core::Api
{
    class OpenPlxContextInternal;

    class OpenPlxContext
    {
        friend class OpenPlxContextInternal;
        public:
            /**
             * @brief Construct a new openplx Context object using OPENPLX_BUNDLE_PATH environment variable for bundle directories
             */
            OpenPlxContext();

            /**
             * @brief Construct a new openplx Context object using provided paths for bundle directories
             */
            explicit OpenPlxContext(const std::vector<std::string>& paths);

            ~OpenPlxContext();

            /**
             * @brief Create bundle paths from OPENPLX_BUNDLE_PATH environment variable
             *
             * @return std::vector<std::string>
             */
            static std::vector<std::string> deriveBundlePaths();

            /**
             * @brief Create bundle paths from provided string
             *
             * @param path a semicolon separated string with the directories containing bundles
             * @return std::vector<std::string>
             */
            static std::vector<std::string> deriveBundlePaths(const std::string& path);

            /**
             * Note that the type of errors that are presented depends on what steps in parse pipeline has been taken
             *
             * @return All errors that has occured, if any
             */
            Errors getErrors();

            bool hasErrors();

            void clearErrors();

            /**
             * Returns all created objects for this context
             */
            std::vector<openplx::Core::ObjectPtr> getRegisteredObjects() const;

            /**
             * Decouples the nodes from the context, preventing the analysis context destructor
             * to deallocate them. The user is from this point responsible for calling `unbind`
             * on all documents to prevent memory leaks.
             */
            void release();

            /**
             * Clears all state from the context retaining the configuration
             */
            void clear();

            void useUuidv5(std::string namespace_id);

            void enableCache();
            void disableCache();
            bool cacheEnabled() const;
            void invalidateCache(std::string source_id);

        private:
            std::unique_ptr<OpenPlxContextInternal> m_internal;

    };
}
