#pragma once

#include <memory>
#include <variant>
#include <vector>
#include <optional>
#include <string>
#include <ostream>

namespace openplx
{
    // Forward declaration
    class Blob;
}

namespace openplx::Core
{
    enum class AnyType {
        Int,
        Real,
        Bool,
        String,
        Blob,
        Object,
        Array,
        Reference,
        Undefined
    };

    // Forward declaration
    class Object;

    /**
     * @brief Common base class for model root class (Object) and primitives
     *
     */
    class Any {
        public:
            Any();
            explicit Any(double value);
            explicit Any(int64_t value);
            explicit Any(bool value);
            explicit Any(std::string value);
            explicit Any(std::vector<Any> value);
            explicit Any(std::optional<double> value);
            explicit Any(std::optional<int64_t> value);
            explicit Any(std::optional<bool> value);
            explicit Any(std::optional<std::string> value);
            explicit Any(std::optional<std::vector<Any>> value);
#ifdef SWIG
            explicit Any(std::shared_ptr<Object> value) {
                if (value == nullptr) {
                    m_type = AnyType::Undefined;
                    m_value = std::shared_ptr<Object>();
                } else {
                    m_type = AnyType::Object;
                    m_value = std::move(value);
                }
            }
            Any::Any(std::shared_ptr<Blob> value);
            explicit Any(std::optional<std::shared_ptr<Object>> value) {
                if (value.has_value()) {
                    auto& value_value = value.value();
                    if (value_value == nullptr) {
                        m_type = AnyType::Undefined;
                        m_value = std::shared_ptr<Object>();
                    } else {
                        m_type = AnyType::Object;
                        m_value = std::move(value_value);
                    }
                } else {
                    m_type = AnyType::Undefined;
                    m_value = std::shared_ptr<Object>();
                }
            }
            Any::Any(std::optional<std::shared_ptr<Blob>> value);
#else
            template <class T>
            explicit Any(std::shared_ptr<T> value) {
                if (value == nullptr) {
                    m_type = AnyType::Undefined;
                    m_value = std::shared_ptr<Object>();
                } else {
                    m_type = AnyType::Object;
                    m_value = std::shared_ptr<Object>(std::move(value));
                }
            }
            explicit Any(std::weak_ptr<Object> reference);

            template <class T>
            explicit Any(std::optional<std::shared_ptr<T>> value) {
                if (value.has_value()) {
                    auto& value_value = value.value();
                    if (value_value == nullptr) {
                        m_type = AnyType::Undefined;
                        m_value = std::shared_ptr<Object>();
                    } else {
                        m_type = AnyType::Object;
                        m_value = std::shared_ptr<Object>(std::move(value_value));
                    }
                } else {
                    m_type = AnyType::Undefined;
                    m_value = std::shared_ptr<Object>();
                }
            }
            explicit Any(std::optional<std::weak_ptr<Object>> reference);
#endif

            Any(const Any& p) = default;
            Any(Any&&) = default;

            Any& operator=(const Any&) = default;
            Any& operator=(Any&&) = default;
            bool operator==(const Any& rhs) const;
            bool operator!=(const Any& rhs) const;

            static Any fromString(std::string value);
            static Any fromArray(std::vector<Any> value);
            static Any fromObject(std::shared_ptr<Object> value);

            inline const char * typeAsString() const {return Any::typeToString(this->m_type);}
            static const char * typeToString(const AnyType& t);


            AnyType getType() const;


            bool isReal() const;

            bool isInt() const;

            /* Int or Real */
            bool isNumber() const;

            bool isBool() const;

            bool isString() const;

            bool isBlob() const;

            bool isObject() const;

            bool isArray() const;

            bool isReference() const;

            bool isUndefined() const;

            /**
             * @brief returns the Real value
             * @throws RuntimeException if Any does not represent Real
             * @return double representing the Real
             */

            double asReal() const;

            /**
             * @brief returns the Int value
             * @throws RuntimeException if Any does not represent Int
             * @return int64_t representing the Int
             */

            int64_t asInt() const;

            /**
             * @brief returns the Bool value
             * @throws RuntimeException if Any does not represent Bool
             * @return bool representing the Bool
             */

            bool asBool() const;

            /**
             * @brief returns the String value
             * @throws RuntimeException if Any does not represent String
             * @return std::string representing the String
             */

            std::string asString() const;

            /**
             * @brief returns the Blob value
             * @throws RuntimeException if Any does not represent Blob
             * @return std::shared_ptr<Blob> representing the Blob
             */
            std::shared_ptr<Blob> asBlob() const;

            /**
             * @brief returns the Object value
             * @throws RuntimeException if Any does not represent Object
             * @return std::shared_ptr<Object> representing the Object
             */

            std::shared_ptr<Object> asObject() const;

            std::weak_ptr<Object> asReference() const;

            /**
             * @brief returns the Array value
             * @throws RuntimeException if Any does not represent Array
             * @return const std::vector<Any>& representing the Array
             */

            const std::vector<Any>& asArray() const;

            friend std::ostream& operator<<(std::ostream& os, const Any& any);
        private:
            AnyType m_type;
            std::variant<double, int64_t, bool, std::string, std::shared_ptr<Blob>, std::vector<Any>,
                         std::shared_ptr<Object>, std::weak_ptr<Object>> m_value;
    };

#ifndef SWIG
    template<>
    Any::Any(std::shared_ptr<Blob> value);
#endif
}
