/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Vec3.h>
#include <agxCollide/BoundingAABB.h>

namespace agxTerrain
{
  /*
  This class deals with position-index conversions INSIDE the voxel grid.
  A position in the grid, aka grid position or voxel space position, is not the
  same as a local position in the terrain, which has it origo in the middle of
  the terrain. The voxel space origo is situated in the "bottomleft" corner of
  the terrain, coinciding with the terrain index (0, 0).
  */
  class VoxelGridUtils
  {
  public:
    /*
    \param gridPosition - The position in voxel space.
    \param voxelSize - The side length of a voxel in the grid.
    \return the voxel index of the specified voxel space position.
    */
    static inline agx::Vec3i getVoxelIndexFromGridPosition(agx::Vec3 gridPosition, agx::Real voxelSize)
    {
      gridPosition *= (1.0 / voxelSize);
      return agx::Vec3i(  (agx::Int)std::floor(gridPosition.x() + 0.5),
                          (agx::Int)std::floor(gridPosition.y() + 0.5),
                          (agx::Int)std::floor(gridPosition.z() + 0.5)  );
    }

    /*
    \param voxelIndex - the index of a voxel in grid.
    \param voxelSize - The side length of a voxel in the grid.
    \return the center position of the specified voxel in voxel space coordinates.
    */
    static inline agx::Vec3 getGridPositionFromVoxelIndex(agx::Vec3i voxelIndex, agx::Real voxelSize)
    {
      return agx::Vec3( static_cast<agx::Real>((voxelIndex.x()))* voxelSize,
                        static_cast<agx::Real>((voxelIndex.y()))* voxelSize,
                        static_cast<agx::Real>((voxelIndex.z()))* voxelSize  );
    }

    /**
    Transforms a world bound to grid given grid transform and additional/extended size.
    \param bound - bound in world.
    \param gridInvTransform - transform of the grid local space.
    \return the transformed bound in the grid coordinate system.
    */
    static inline agx::Bound3 transformBound( const agx::Bound3& bound,
                                              const agx::AffineMatrix4x4& gridInvTransform,
                                              agx::Real extend = agx::Real( 0 ) )
    {
      return agxCollide::BoundingAABB{ agx::Bound3{ bound.min() - agx::Vec3( extend ),
                                                    bound.max() + agx::Vec3( extend ) },
                                       gridInvTransform };
    }
  };
}
