/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agxCollide/Geometry.h>

#include <agxTerrain/GroundCollapseUtils.h>
#include <agxTerrain/CollapseSettings.h>
#include <agxTerrain/GroundCollapseResult.h>
#include <agxTerrain/GroundCollapseCalculator.h>

#include <agxTerrain/export.h>


namespace agxTerrain
{
  namespace GroundCollapse
  {
    AGX_DECLARE_POINTER_TYPES(GroundCollapseController);

    /**
    Control ground collapse functionality in the terrain. Ground collapse needs
    to be enabled beforehand via TerrainProperties::setEnableGroundCollapse.

    After ground collapse is enabled and ground collapse zones are designated via
    \ref addGroundCollapseZone, calculations will be performed in each time step whether
    external loads on the soil, from a bulldozer for example, causes the ground to
    collapse.

    The ground collapse simulations can be configured via the related settings classes,
    obtained via \ref getSettings, \ref getRenderSettings and \ref getSlopeSettings.
    */
    class AGXTERRAIN_EXPORT GroundCollapseController : public agx::Referenced
    {

    public:
      /*
      Add a zone where the ground collapse algorithm should be run.
      \param zone - a geometry designating the zone where to run ground collapse algorithm. Must be overlapping the corresponding terrain.
      */
      void addGroundCollapseZone(agxCollide::Geometry* zone);

      /*
      Remove all the added collapse zones. This effectively disables ground collapse functionality.
      */
      void clearGroundCollapseZones();

      /*
      Get the geometry designating the zone where ground collapse algorithm is run every timestep.
      */
      agxCollide::GeometryPtrVector getGroundCollapseZones();

      /**
      Get the terrain contacts that will be used in ground collpase calculations
      */
      TerrainContactPtrVector getCollapseContacts();

      /*
      Get the terrain contacts inside the given geometry.
      */
      TerrainContactPtrVector getContactsInGroundCollapseZone();

      /*
      Get the settings for the ground collapse algorithm.
      */
      inline GroundCollapse::CollapseSettings* getSettings() const { return m_settings; };

      /*
      Set the settings for the ground collapse algorithm.
      \param settings - the settings to use.
      */
      void setSettings( GroundCollapse::CollapseSettings* settings );

      /*
      Get the render settings for the ground collapse algorithm.
      */
      inline GroundCollapse::CollapseRenderSettings* getRenderSettings() const { return m_renderSettings; };

      /*
      Set the render settings for the ground collapse algorithm.
      \param renderSettings - the render settings to use.
      */
      void setRenderSettings( GroundCollapse::CollapseRenderSettings* renderSettings );

      /*
      Get the settings for the slope detection algorithm.
      */
      inline GroundCollapse::SlopeDetectionSettings* getSlopeSettings() const { return m_slopeSettings; };

      /*
      Set the settings for the slope detection algorithm.
      \param settings - the settings to use.
      */
      inline void setSlopeSettings(GroundCollapse::SlopeDetectionSettings* slopeSettings) { m_slopeSettings = slopeSettings; };

      /*
      Get the results of the ground collapse algorithm in this timestep.
      Should always be set in the users simulation post-step.
      See GroundCollapse::GroundCollapseResult for more information.
      */
      GroundCollapse::GroundCollapseResult* getCollapseResult() const;

      /*
      Add a user designated FailureSurface to the ground collapse simulations.
      Will remain in the simulation until removed by the user.
      \param customSurface - A custom failure surface provided by the user.
      \return true if the customSurface could be added, false if the surface is
      not valid and therfore could not be added.
      */
      bool addCustomFailureSurface( FailureSurface* customSurface );

      /*
      \return a vector of all the added custom failure surfaces.
      */
      FailureSurfacePtrVector getCustomFailureSurfaces() const;

      /*
      Remove all user designated custom failure surfaces.
      */
      void clearCustomSurfaces();

      DOXYGEN_START_INTERNAL_BLOCK()

    // Internal methods:
    public:
      /*
      Construct controller for a specific terrain.
      */
      GroundCollapseController(agxTerrain::Terrain* terrain);

      /*
      For serialization
      */
      GroundCollapseController();

      /*
      Start the ground collapse algorithm inside the added collapse zone.
      Sets the GroundCollapseResult which can be fetched via getCollapseResult().
      Must be run after the terrain has run it's post step.
      */
      void onPost();

      Terrain* getTerrain() const { return m_terrain; };

      int findLargestFailedSurface( const  FailureSurfaceResultVector& results );

      AGXTERRAIN_STORE_RESTORE_INTERFACE;

    protected:
      ~GroundCollapseController() {};

    private:
      Terrain*                                 m_terrain;
      CollapseSettingsRef                      m_settings;
      CollapseRenderSettingsRef                m_renderSettings;
      SlopeDetectionSettingsRef                m_slopeSettings;
      GroundCollapseCalculatorRef              m_calculator;
      agxCollide::GeometryRefVector            m_zones;
      GroundCollapse::GroundCollapseResultRef  m_collapseResult;
      FailureSurfaceRefVector                  m_customSurfaces;
      DOXYGEN_END_INTERNAL_BLOCK()

    };

  }
}
