/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agx/Referenced.h>
#include <agxStream/Serializable.h>
#include <agxStream/InputArchive.h>
#include <agxStream/OutputArchive.h>
#include <agx/config/AGX_USE_AGXTERRAIN.h>
#include <agxTerrain/export.h>

namespace agxTerrain
{
  namespace GroundCollapse
  {
    AGX_DECLARE_POINTER_TYPES(FailureSurface);
    AGX_DECLARE_VECTOR_TYPES(FailureSurface);

    /*
    Class that describes a failure surface used in the ground collapse algorithm.

    A failure surface is a sphere that intersects points x_a and x_b on the terrain surface, and has a certain radius.
    The radiusNumber, aNumber and bNumber are integers >= 0 that identify which failure surface it is in the
    configuration of failure surfaces used in a ground collapse algorithm.
    */
    class AGXTERRAIN_EXPORT FailureSurface : public agx::Referenced, public agxStream::Serializable
    {
    public:
      FailureSurface(const agx::Vec3& upDirection,
                     const agx::Vec3& xa,
                     const agx::Vec3& xb,
                     agx::UInt radiusNumber,
                     agx::UInt aNumber=0,
                     agx::UInt bNumber=0 );

      /*
      Get the center position of the failure surface.
      */
      inline agx::Vec3 getPosition() const { return m_position; };

      /*
      Get the radius of the failure surface.
      */
      inline agx::Real getRadius() const { return m_radius; };

      /*
      Get the intersection point of the failure surface on/below the slope.
      */
      inline agx::Vec3 getXa() const { return m_xa; };

      /*
      Get the intersection point of the failure surface above the slope.
      */
      inline agx::Vec3 getXb() const { return m_xb; };

      /*
      \return the total outer area of the sphere that creates the failure surface.
      */
      inline agx::Real getArea() const { return agx::Real( 4.0 ) * agx::PI * std::pow( getRadius(), 2 ); };
      
      /*
      Check that the failure surface is defined correctly; that xA and xB are not the same and that the radius is greater than 0.
      */
      bool isValid();
      

      DOXYGEN_START_INTERNAL_BLOCK()
      AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::GroundCollapse::FailureSurface)
      inline agx::Vec3 getRadiusVector() const { return m_radiusVector; };
      inline agx::UInt getRadiusNumber() const { return m_radiusNumber; };
      inline agx::UInt getANumber() const { return m_ANumber; };
      inline agx::UInt getBNumber() const { return m_BNumber; };
      FailureSurface();
      DOXYGEN_END_INTERNAL_BLOCK()

    protected:
      virtual ~FailureSurface();

    private:
      agx::Vec3                   m_xa;
      agx::Vec3                   m_xb;
      agx::UInt                   m_radiusNumber;

      agx::UInt                   m_ANumber;
      agx::UInt                   m_BNumber;

      agx::Vec3                   m_radiusVector;

      agx::Vec3                   m_position;
      agx::Real                   m_radius;
    };
  }
}