/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SystemNodeProxy.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( ITriaxialSignalSystemNode );
  AGX_DECLARE_VECTOR_TYPES( ITriaxialSignalSystemNode );
  AGX_DECLARE_POINTER_TYPES( TriaxialSignalSystemNode );
  AGX_DECLARE_VECTOR_TYPES( TriaxialSignalSystemNode );

  /**
  Interface for types carrying a triaxial signal.
  */
  class AGXSENSOR_EXPORT ITriaxialSignalCarrier
  {
    public:
      /**
      \return the triaxial signal that this type is carrying
      */
      virtual const std::vector<agx::Vec3>& getTriaxialSignal() const = 0;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~ITriaxialSignalCarrier() = default;

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  /**
  Interface for implementation details for system nodes carrying a triaxial signal.
  */
  class AGXSENSOR_EXPORT ITriaxialSignalSystemNode : public ISystemNodeProxy
  {
    public:
      /**
      \return new, unique, triaxial signal system node (proxy) of this implementation
      */
      virtual TriaxialSignalSystemNodeRef createTriaxialNodeProxy() = 0;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual SystemNodeProxyRef createProxy() override final;

      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::ITriaxialSignalSystemNode );

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  /**
  A system node in the sensor processing tree carrying a triaxial signal and an implementation for
  how to process it.
  */
  class AGXSENSOR_EXPORT TriaxialSignalSystemNode : public SystemNodeProxy,
                                                    public ITriaxialSignalCarrier
  {
    public:
      /**
      Construct a triaxial signal system node instance with the given \p implementation.
      \param implementation - behavioral implementation for this proxy (should not be nullptr)
      */
      TriaxialSignalSystemNode( ISystemNodeProxy* implementation );

      /**
      \return the triaxial signal that this node is carrying
      */
      std::vector<agx::Vec3>& getTriaxialSignal();

      /**
      \return the triaxial signal that this node is carrying
      */
      virtual const std::vector<agx::Vec3>& getTriaxialSignal() const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      TriaxialSignalSystemNode();

      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::TriaxialSignalSystemNode );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      std::vector<agx::Vec3> m_triaxialSignal;
  };
}
