/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/BinaryOutputView.h>

#include <agx/Logger.h>

namespace agxSensor
{
  /**
  Common interface for sensor output data.
  */
  class AGXSENSOR_EXPORT ISensorOutput
  {
    public:
      /**
      Checks whether sizeof(T) == elementSize and issues a warning if it is not.
      \return true if the size matches, otherwise false
      */
      template<typename T>
      static bool verifySize( size_t elementSize );

    public:
      /**
      Default constructor.
      */
      ISensorOutput() = default;

      /**
      "Element size" is the summed data size of the currently added sensor output fields.
      \return the current, summed, data size of the added fields
      */
      virtual size_t getElementSize() const = 0;

      /**
      View over data. The size of type T has to match the element size of the added sensor output
      fields. If the size doesn't match, an empty view is returned.
      This call blocks until the sensor data is fully assembled and available.
      \return view of data as type T
      */
      template<typename T>
      BinaryOutputView<T> view();

      /**
      Returns access to the sensor output data.
      This call blocks until the sensor data is fully assembled and available.
      \return sensor output data as a pure binary buffer
      */
      virtual const BinaryOutputBuffer& getData() = 0;

      /**
      \param markAsRead - true to reset unread state, resulting in hasUnreadData returning false
                          until new data is available. False to not mark the current data as read.
      \return true when the data has been updated but not read/used/fetched by the user, false if
              the data has been read
      */
      virtual bool hasUnreadData( bool markAsRead = true ) = 0;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~ISensorOutput() = default;

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  template<typename T>
  bool ISensorOutput::verifySize( size_t elementSize )
  {
    const auto isValidSize = sizeof( T ) == elementSize;
    if ( !isValidSize ) {
      LOGGER_WARNING() << "View data type size mismatch, sizeof(" << typeid( T ).name() << ") = "
                       << sizeof( T ) << " != " << elementSize << " (elementSize)."
                       << LOGGER_ENDL();
    }
    return isValidSize;
  }

  template<typename T>
  BinaryOutputView<T> ISensorOutput::view()
  {
    if ( !ISensorOutput::verifySize<T>( this->getElementSize() ) )
      return {};
    return getData().view<T>();
  }
}
