/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/LidarModelMechanicalRotation.h>
#include <agxSensor/LidarRayPatternGenerator.h>
#include <agxSensor/LidarRayPatternHorizontalSweep.h>

namespace agxSensor
{
  /**
  Simulation implementation of the Ouster OSx lidar model series - OS0, OS1, OS2.
  Use by instantiating the derived classes with default or user selected settings.
  Range and fov is overridable from default settings, useful for instance if simulating
  an older / newer hardware revision.
  */
  class AGXSENSOR_EXPORT LidarModelOusterOS : public LidarModelMechanicalRotation
  {
  public:
    // Ouster OSx hardware configuration
    enum ChannelCount { ch_32, ch_64, ch_128 };
    enum BeamSpacing { Uniform, AboveHorizon, BelowHorizon };

    // Ouster OSx software settings
    enum LidarMode { Mode_512x10, Mode_512x20, Mode_1024x10, Mode_1024x20, Mode_2048x10 };

  public:
    /*
    \return lidar channel count setting
    */
    ChannelCount getChannelCount() const { return m_channelCount; }

    /*
    \return lidar channel distribution setting
    */
    BeamSpacing getBeamSpacing() const { return m_beamSpacing; }

    /*
    \return lidar channel mode
    */
    LidarMode getLidarMode() const { return m_lidarMode; }

    /**
    Base Ouster OS lidar class. NOTE: not intended to be instantiated, use the derived classes.
    \param verticalFov - veritcal field of view in radians x (horizontal) and y (vertical)
    \param resolution - resolution in radians x (horizontal) and y (vertical)
    \param range - lidar ray range
    \param frequency - number of revolutions per second
    \param properties - additional lidar properties (default is null)
    */
    LidarModelOusterOS(
      agx::RangeReal32 verticalFov,
      agx::Vec2 resolution,
      agx::RangeReal32 range,
      agx::Real frequency,
      LidarProperties* properties = nullptr);

    LidarModelOusterOS();

    virtual ~LidarModelOusterOS() = default;

    AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarModelOusterOS );

  protected:
    static agx::Real frequencyValue(LidarMode lidarMode);
    static agx::RangeReal32 verticalFOVValue(agx::Real verticalRange, BeamSpacing beamSpacing);
    static agx::Vec2 resolutionValue(ChannelCount channelCount, LidarMode lidarMode, agx::RangeReal32 verticalFov);
    static agx::Real horizontalResolutionValue(LidarMode lidarMode);
    static agx::Real channelCountValue(ChannelCount channelCount);

    static agx::Real rangeFactorFromLidarMode( LidarMode lidarMode );

    ChannelCount m_channelCount;
    BeamSpacing m_beamSpacing;
    LidarMode m_lidarMode;
  };

  /**
  Specialized class for the Ouster OS0 lidar model.
  */
  class AGXSENSOR_EXPORT LidarModelOusterOS0 : public LidarModelOusterOS
  {
  public:
    /**
    Ouster OS0 - Ultra-Wide View High-Resolution Imaging Lidar (hardware revision 7)
    Constructor with settings. 
    \param channelCount - configuration option for vertical resolution. Options are 32, 64, 128
    \param beamSpacing - configuration option for vertical laser distribution. Uniform is the whole fov, above / below horizon collects all channels on one half of the fov
    \param lidarMode - the combination of horizontal resolution and rotation frequency settings. 
    \param verticalRange - the vertical field of view. It is suggested to use default value.
    \param lidarRange - lidar ray range. It is suggested to use default value.
    */
    LidarModelOusterOS0(
      ChannelCount channelCount = ChannelCount::ch_64,
      BeamSpacing beamSpacing = BeamSpacing::Uniform,
      LidarMode lidarMode = LidarMode::Mode_1024x10,
      agx::Real verticalRange = 90.0,
      agx::RangeReal32 lidarRange = { 0.5, 75.0 }
    );

    AGXSTREAM_DECLARE_SERIALIZABLE(agxSensor::LidarModelOusterOS0);
  };

  /**
  Specialized class for the Ouster OS1 lidar model.
  */
  class AGXSENSOR_EXPORT LidarModelOusterOS1 : public LidarModelOusterOS
  {
    public:
      /**
      Ouster OS1 - Mid-Range High-Resolution Imaging Lidar (hardware revision 7)
      Constructor with settings.
      \param channelCount - configuration option for vertical resolution. Options are 32, 64, 128
      \param beamSpacing - configuration option for vertical laser distribution. Uniform is the whole fov, above / below horizon collects all channels on one half of the fov
      \param lidarMode - the combination of horizontal resolution and rotation frequency settings. 
      \param verticalRange - the vertical field of view. It is suggested to use default value.
      \param lidarRange - lidar ray range. It is suggested to use default value.
      */
      LidarModelOusterOS1(
        ChannelCount channelCount = ChannelCount::ch_64,
        BeamSpacing beamSpacing = BeamSpacing::Uniform,
        LidarMode lidarMode = LidarMode::Mode_1024x10,
        agx::Real verticalRange = 45.0,
        agx::RangeReal32 lidarRange = { 0.5, 170.0 }
      );

      AGXSTREAM_DECLARE_SERIALIZABLE(agxSensor::LidarModelOusterOS1);
  };

  /**
  Specialized class for the Ouster OS2 lidar model.
  */
  class AGXSENSOR_EXPORT LidarModelOusterOS2 : public LidarModelOusterOS
  {
    public:
      /**
      Ouster OS2 - Long Range High-Resolution Imaging Lidar (hardware revision D)
      Constructor with settings.
      \param channelCount - configuration option for vertical resolution. Options are 32, 64, 128
      \param beamSpacing - configuration option for vertical laser distribution. Uniform is the whole fov, above / below horizon collects all channels on one half of the fov
      \param lidarMode - the combination of horizontal resolution and rotation frequency settings. 
      \param verticalRange - the vertical field of view. It is suggested to use default value.
      \param lidarRange - lidar ray range. It is suggested to use default value.
      */
      LidarModelOusterOS2(
        ChannelCount channelCount = ChannelCount::ch_64,
        BeamSpacing beamSpacing = BeamSpacing::Uniform,
        LidarMode lidarMode = LidarMode::Mode_1024x10,
        agx::Real verticalRange = 22.5,
        agx::RangeReal32 lidarRange = { 1.0, 210.0 }
      );

      AGXSTREAM_DECLARE_SERIALIZABLE(agxSensor::LidarModelOusterOS2);
  };
}
