/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/Referenced.h>
#include <agx/HashVector.h>

namespace agx
{
  /**
  Holds references to instances given common or unique type. It's explicit in type
  when it's up to the user to pick granularity, e.g., handler.add<agxCollide::Shape>
  vs. handler.add<agxCollide::Box>, handler.add<agxCollide::Cylinder>...

    ReferenceHandler refHandler;
    auto box = new agxCollide::Box( he );
    refHandler.add<agxCollide::Shape>( box );    // ref count 1
    refHandler.contains<agxCollide::Box>();      // false, type Box hasn't been associated.
    refHandler.add<agxCollide::Box>( box );      // ref count 2
    refHandler.remove<agxCollide::Shape>( box ); // ref count 1
    ...
  */
  class ReferencedHandler
  {
    public:
      using RefRef = agx::ref_ptr<agx::Referenced>;
      using RefTable = agx::HashVector<const agx::Referenced*, RefRef>;
      using TypeRefTable = agx::HashTable<size_t, RefTable>;

    public:
      template<typename T>
      bool contains( const agx::Referenced& instance ) const
      {
        return this->template contains<T>() &&
               this->template get<T>().contains( &instance );
      }

      template<typename T>
      bool contains() const
      {
        return m_typeRefTable.contains( this->template id<T>() );
      }

      template<typename T>
      size_t size() const
      {
        return this->template get<T>().size();
      }

      template<typename T>
      size_t id() const
      {
        static constexpr auto& type = typeid( T );
        return type.hash_code();
      }

      template<typename T>
      bool add( agx::Referenced& instance )
      {
        if ( this->template contains<T>( instance ) )
          return false;

        this->template get<T>().insert( &instance, &instance );

        return true;
      }

      template<typename T>
      bool remove( agx::Referenced& instance )
      {
        if ( !this->template contains<T>() )
          return false;

        auto& table = this->template get<T>();
        auto success = table.erase( &instance );
        if ( success && table.empty() )
          m_typeRefTable.erase( this->template id<T>() );

        return success;
      }

      template<typename T, typename VisitorT>
      void visit( VisitorT&& visitor ) const
      {
        const auto& table = this->template get<T>();
        for ( const auto& pp : table )
          visitor( *(pp.first->template asSafe<T>()) );
      }

      template<typename T>
      void clear()
      {
        if ( !this->template contains<T>() )
          return;

        this->template get<T>().clear();
        m_typeRefTable.erase( this->template id<T>() );
      }

      void clear()
      {
        m_typeRefTable.clear();
      }

    private:
      template<typename T>
      RefTable& get()
      {
        return this->get( this->template id<T>() );
      }

      template<typename T>
      const RefTable& get() const
      {
        return this->get( this->template id<T>() );
      }

      RefTable& get( size_t id )
      {
        return m_typeRefTable[ id ];
      }

      const RefTable& get( size_t id ) const
      {
        const static RefTable emptyTable{};
        const auto it = m_typeRefTable.find( id );
        return it != m_typeRefTable.end() ? it->second : emptyTable;
      }

    private:
      TypeRefTable m_typeRefTable;
  };
}
