#pragma once

#include <memory>
#include <string>
#include <vector>
#include <utility>
#include <filesystem>
#include <openplx/AnalysisContext.h>
#include <openplx/Any.h>
#include <openplx/OpenPlxContext.h>
#include <openplx/Node.h>
#include <openplx/EvaluatorContext.h>
#include <openplx/Plugin.h>

namespace openplx::Core::Api {

    /**
     * @brief Loads and evaluates a .openplx file and all dependant bundles
     * @param openplx_file
     * @param modelname the model to evaluate, if not provided then the last model will be used
     * @param openplx_context the context to evaluate in
     * @return openplx::Core::ObjectPtr an evaluated model tree or nullptr if
     *         evaluation failed.
     */
    openplx::Core::ObjectPtr
    loadModelFromFile(const std::filesystem::path& openplx_file, std::optional<std::string> modelname,
                      OpenPlxContext& openplx_context);

    /**
     * @brief Loads and evaluates a model from string
     *
     * @param openplx_string the string containing one or more models
     * @param modelname the model to evaluate, if not provided then the last model will be used
     * @param openplx_context the context to evaluate in
     * @return openplx::Core::ObjectPtr an evaluated model tree or nullptr if
     *         evaluation failed.
     */
    openplx::Core::ObjectPtr
    loadModelFromString(const std::string& openplx_string, std::optional<std::string> modelname,
                        OpenPlxContext& openplx_context);

    /**
     * @brief Loads and evaluates a model from a openplx document
     *
     * @param document a openplx document
     * @param modelname the model to evaluate, if not provided then the last model in the document will be used
     * @param openplx_context the context to evaluate in
     * @return openplx::Core::ObjectPtr an evaluated model tree or nullptr if evaluation failed.
     */
    openplx::Core::ObjectPtr
    loadModelFromDocument(const DocPtr& document, std::optional<std::string> modelname,
                          OpenPlxContext& openplx_context);

    /**
     * @brief Run the semantic analyzer, and then evaluate the modelname in the OpenPLX context provided, returning the evaluated tree for that modelname
     * @param openplx_context
     * @param modelname the modelname to evaluate, if nullopt the last model in the <default> bundle will be evaluated
     * @return The evaluated tree
     */
    openplx::Core::ObjectPtr
    analyzeAndEvaluate(OpenPlxContext &openplx_context, const DocPtr& document, std::optional<std::string> modelname);

    /**
     * @brief Run the semantic analyzer, updating OpenPLX context
     * @param openplx_context
     * @return
     */
    std::shared_ptr<Analysis::AnalysisContext> analyze(OpenPlxContext& openplx_context, ModelDeclPtr model);

    /**
     * @brief Parse all dependent files given a .openplx file
     * @return The document coordinate and a list of all parsed documents
     */
    std::pair<DocPtr, std::vector<DocPtr>>
    parseFromFile(const std::filesystem::path& openplx_file, OpenPlxContext& openplx_context);

    /**
     * @brief Parse all dependent files given a .openplx file but replace
     *        that specific file with new content. Useful for Language Servers.
     * @return The document coordinate and a list of all parsed documents
     */
    std::pair<DocPtr, std::vector<DocPtr>>
    parseFromFileWithVirtualContent(
        const std::filesystem::path& openplx_file,
        const std::string& content,
        OpenPlxContext& openplx_context);

    /**
     * @brief Parse all dependent files given a .openplx file
     * @return The document coordinate and a list of all parsed documents
     */
    std::pair<DocPtr, std::vector<DocPtr>>
    parseFromString(const std::string& openplx_string, OpenPlxContext& openplx_context);

    /**
     * @brief Register a plugin that will parse non-openplx files to openplx AST
     * allowing them to be referenced from .openplx files automatically.
     */
    void registerPlugin(std::shared_ptr<openplx::Plugin> plugin, OpenPlxContext& openplx_context);

    /**
     * @brief Retrieve all registered plugins.
     */
    std::vector<std::shared_ptr<openplx::Plugin>> getRegisteredPlugins(OpenPlxContext& openplx_context);

    /**
     * @brief Returns the version of openplx.core
     */
    std::string getOpenPlxCoreVersion();
}
