/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/LidarRayPatternGenerator.h>

namespace agxSensor
{
  /**
  Horizontal sweep pattern generating transforms in a sweeping pattern given field of view and
  resolution. The frequency, number of sweeps per second, affects the number of active rays during
  the time step size of the environment.
  */
  class AGXSENSOR_EXPORT LidarRayPatternHorizontalSweep : public LidarRayPatternGenerator
  {
    public:
      /**
      Default constructor, zero rays.
      */
      LidarRayPatternHorizontalSweep();

      /**
      Construct given field of view \p fov (radians x and y, centered about the lidar x-axis),
      angle resolution (radians x and y) \p angleResolution and sweep frequency \p frequency
      (Hz).
      \param fov - field of view in radians, x (horizontal) and y (vertical), centered about the
             lidar x-axis.
      \param angleResolution - angle resolution in radians, x (horizontal) and y (vertical)
      \param frequency - number of sweeps per second
      */
      LidarRayPatternHorizontalSweep( const agx::Vec2& fov, const agx::Vec2& angleResolution,
                                      agx::Real frequency );

      /**
      Construct given field of view \p fov (radians x and y, centered about the lidar x-axis),
      resolution (beam count x and y) \p resolution and sweep frequency \p frequency
      (Hz).
      \param fov - field of view in radians, x (horizontal) and y (vertical), centered about the
             lidar x-axis.
      \param resolution - resolution in number of beams, x (horizontal) and y (vertical)
      \param frequency - number of sweeps per second
      */
      LidarRayPatternHorizontalSweep( const agx::Vec2& fov, const agx::Vec2u& resolution,
                                      agx::Real frequency );

      /**
      Construct given field of view \p horizontal and vertical fov (radians start and stop), angle
      resolution (radians x and y) \p angleResolution and sweep frequency \p frequency (Hz).
      \param horizontalFov - horizontal field of view in radians, x (start) and y (stop)
      \param verticalFov - vertical field of view in radians, x (start) and y (stop)
      \param angleResolution - angle resolution in radians, x (horizontal) and y (vertical)
      \param frequency - number of sweeps per second
      */
      LidarRayPatternHorizontalSweep( const agx::RangeReal& horizontalFov,
                                      const agx::RangeReal& verticalFov,
                                      const agx::Vec2& angleResolution,
                                      agx::Real frequency );

      /**
      Construct given field of view \p horizontal and vertical fov (radians start and stop),
      resolution (beam count x and y) \p resolution and sweep frequency \p frequency (Hz).
      \param horizontalFov - horizontal field of view in radians, x (start) and y (stop)
      \param verticalFov - vertical field of view in radians, x (start) and y (stop)
      \param resolution - resolution in number of beams, x (horizontal) and y (vertical)
      \param frequency - number of sweeps per second
      */
      LidarRayPatternHorizontalSweep( const agx::RangeReal& horizontalFov,
                                      const agx::RangeReal& verticalFov,
                                      const agx::Vec2u& resolution,
                                      agx::Real frequency );

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~LidarRayPatternHorizontalSweep() = default;
      virtual LidarRayPatternInterval getNextInterval( agx::Real dt ) override;
      virtual size_t getNumRays() const override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarRayPatternHorizontalSweep );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      struct PatternState
      {
        agx::Real elapsedTime = 0.0;
        agx::Real sweepTime = 0.0;
        size_t last = std::numeric_limits<size_t>::max();
      };

    private:
      void setRays( const agx::RangeReal& horizontalFov, const agx::RangeReal& verticalFov,
                    const agx::Vec2u& resolution );

    private:
      PatternState m_state;
      std::shared_ptr<LidarRayPatternStorage> m_pattern;
  };
}
