/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/export.h>

#include <agx/agx.h>

namespace agxSensor
{
  /**
  View for reading sections of a binary packed buffer of the specified template parameter type.
  */
  template<typename T>
  class BinaryOutputView
  {
    public:
      using value_type = T;

      struct iterator
      {
        using difference_type = std::ptrdiff_t;
        using value_type = T;
        using pointer = T*;
        using reference = T&;
        using iterator_category = std::random_access_iterator_tag;
      };

    public:
      /**
      Constructs a default empty binary output view.
      */
      inline BinaryOutputView() = default;

      /**
      Constructs a binary output view starting from the specified buffer address and extending in 
      the specified size.
      \param buffer - address from where the binary output view should start
      \param size - size of the binary output view window, in number of template parameter types
      */
      BinaryOutputView( const void* buffer, size_t size );

      /**
      \return size of the buffer view section, in number of template parameter types
      */
      size_t size() const;

      /**
      \return true if empty, false otherwise
      */
      bool empty() const;

      /**
      \return beginning iterator
      */
      const T* begin() const;

      /**
      \return ending iterator
      */
      const T* end() const;

      /**
      Random read access of value at the specified index.
      \param localIndex - index within the buffer view region
      \return read access to the element
      */
      const T& operator[]( size_t localIndex ) const;

      DOXYGEN_START_INTERNAL_BLOCK()

    private:
      const T* m_buffer = nullptr;
      size_t m_size = 0u;

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  /**
  Buffer used for sensor output of a wide range of output element types.
  */
  class AGXSENSOR_EXPORT BinaryOutputBuffer
  {
    public:
      /**
      \return size of the buffer in number of elements
      */
      size_t size() const;

      /**
      \return element size in bytes
      */
      size_t elementSize() const;

      /**
      \return true if empty, false otherwise
      */
      bool empty() const;

      /**
      Resize the buffer to fit the given number of elements of the given size.
      \param size - number of elements to fit in the buffer
      \param elementSize - size of one element in bytes
       */
      void resize( size_t size, size_t elementSize );

      /**
      \return native raw address to the start of the buffer
      */
      void* rwPtr();

      /**
      \return native raw address to the start of the buffer
      */
      const void* rwPtr() const;

      /**
      \return address to the start of the buffer in the given template argument type
      */
      template<typename T>
      const T* ptr() const;

      /**
      Create a view into a section of the buffer.
      \param startIndex - starting index position within the buffer in number of elements
      \param numElements - size of the view section in number of elements
      \return Read view into a section of the buffer
      */
      template<typename T>
      BinaryOutputView<T> view( size_t startIndex = 0u, 
                                size_t numElements = agx::InvalidIndex ) const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      BinaryOutputBuffer();
      ~BinaryOutputBuffer();

      BinaryOutputBuffer(const BinaryOutputBuffer&) = delete;

    private:
      char* m_buffer;
      size_t m_size;
      size_t m_elementSize;

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  template<typename T>
  inline BinaryOutputView<T>::BinaryOutputView( const void* buffer, size_t size )
    : m_buffer( (const T*)buffer )
    , m_size( size )
  {
  }

  template<typename T>
  inline size_t BinaryOutputView<T>::size() const
  {
    return m_size;
  }

  template<typename T>
  inline bool BinaryOutputView<T>::empty() const
  {
    return m_size == 0u;
  }

  template<typename T>
  inline const T* BinaryOutputView<T>::begin() const
  {
    return m_buffer;
  }

  template<typename T>
  inline const T* BinaryOutputView<T>::end() const
  {
    return m_buffer + m_size;
  }

  template<typename T>
  inline const T& BinaryOutputView<T>::operator[]( size_t localIndex ) const
  {
    return m_buffer[ localIndex ];
  }

  inline size_t BinaryOutputBuffer::size() const
  {
    return m_size;
  }

  inline size_t BinaryOutputBuffer::elementSize() const
  {
    return m_elementSize;
  }

  inline bool BinaryOutputBuffer::empty() const
  {
    return m_size == 0u;
  }

  inline void BinaryOutputBuffer::resize( size_t size, size_t elementSize )
  {
    delete[] m_buffer;

    if ( size > 0 && elementSize > 0 )
      m_buffer = new char[ size * elementSize ];
    else
      m_buffer = nullptr;

    m_size = size;
    m_elementSize = elementSize;
  }

  inline void* BinaryOutputBuffer::rwPtr()
  {
    return m_buffer;
  }

  inline const void* BinaryOutputBuffer::rwPtr() const
  {
    return m_buffer;
  }

  template<typename T>
  inline const T* BinaryOutputBuffer::ptr() const
  {
    agxAssert1( sizeof( T ) == m_elementSize, "ptrAs<T> type size mismatch." );
    return (const T*)m_buffer;
  }

  template<typename T>
  inline BinaryOutputView<T> BinaryOutputBuffer::view( size_t startIndex, /*= 0u*/
                                                       size_t numElements /*= agx::InvalidIndex*/ ) const
  {
    if ( m_buffer == nullptr )
      return {};

    if ( startIndex >= m_size )
      return {};

    if ( numElements == agx::InvalidIndex )
      numElements = m_size > startIndex ? m_size - startIndex : 0u;

    return { &(this->template ptr<T>()[ startIndex ]), numElements };
  }

  inline BinaryOutputBuffer::BinaryOutputBuffer()
    : m_buffer( nullptr )
    , m_size( 0u )
    , m_elementSize( 0u )
  {
  }

  inline BinaryOutputBuffer::~BinaryOutputBuffer()
  {
    delete[] m_buffer;
  }

}
