/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/export.h>
#include <agx/Material.h>
#include <agxTerrain/TerrainMaterial.h>
#include <agxTerrain/Shovel.h>



namespace agxTerrain
{
  /**
  Configure the contact material that is to be used in shovel-aggregate contacts. The contact material properties
  are set from the contact material between the shovel geometries and the terrain geometry by default.
  The user can override it by specifying an explicit contact material that is used in the different kinds of
  excavation modes. See 'setShovelAggregateContactMaterial' in agxTerrain::Terrain.

  \param shovelAggregateCM - the shovel-aggregate contact material that should be configured.
  \param contactArea - The area of the shovel-aggregate contact for the specified excavation
                       mode. Used in scaling of Young's modulus.
  \param mode - the excavation mode that is used to search if an explicit contact material has be set for the contact.
                If so, it is used instead of the shovel-terrain contact material.
  \param terrain - The terrain which the aggregate comes from.
  \param shovel - The shovel that interacts with the aggregate.
  \return true if the shovelAggregateCM was configured successfully, false otherwise.
  */
  AGXTERRAIN_EXPORT bool setupShovelAggregateContactMaterial( agx::ContactMaterial* shovelAggregateCM,
                                                              agx::Real contactArea,
                                                              Shovel::ExcavationMode mode,
                                                              Terrain* terrain,
                                                              Shovel* shovel,
                                                              agx::Vec3Vector& points);

  /**
  Set the shovel-aggregate contact material properties from a source contact material that should
  either be from regular shovel-terrain contact material based on a non-homogeneous configuration
  in the terrain.
  \param shovelAggregateCM - the shovel-aggregate contact material where properties will be set.
  \param contactArea - The area of the shovel-aggregate contact for the specified
                       excavation mode. Used in scaling of Young's modulus.
  \param weights - the weights vector used to average the contact materials in the terrain.
  \param terrain - the terrain from which to fetch terrain material properties for the constructed shovel-aggregate contact material.
  \param shovel - the shovel which is interacting with the aggregate.
  */
  AGXTERRAIN_EXPORT void setShovelAggregateContactMaterialProperties( agx::ContactMaterial* shovelAggregateCM,
                                                                      agx::Real contactArea,
                                                                      const agx::RealVector& weights,
                                                                      Terrain* terrain,
                                                                      Shovel* shovel );

  /**
  Set the shovel-aggregate contact material properties from a source contact material that should
  be a user-specified contact material for a given excavation mode.
  \param shovelAggregateCM - the shovel-aggregate contact material where properties will be set.
  \param contactArea - The area of the shovel-aggregate contact for the specified
                       excavation mode. Used in scaling of Young's modulus.
  \param sourceMaterial - the base material whose properties will be used in the setup.
  \param excavationStiffnessMultiplier - The excavation stiffness multiplier to be used in the explicit contacts.
  */
  AGXTERRAIN_EXPORT void setShovelExplicitShovelContactMaterial( agx::ContactMaterial* shovelAggregateCM,
                                                                 agx::Real contactArea,
                                                                 const agx::ContactMaterial* sourceMaterial,
                                                                 agx::Real excavationStiffnessMultiplier);

  AGXTERRAIN_EXPORT void setShovelAggregateContactMaterialFromParticles(agx::ContactMaterial* shovelAggregateCM,
                                                                        agx::Real contactArea,
                                                                        Shovel* shovel);


  /**
  Checks if the specified geometry is valid for creating aggregate contacts against.
  \return true if the specified geometry can be used for aggregate contacts.
  */
  AGXTERRAIN_EXPORT bool isValidContactGeometry( agxCollide::Geometry* geometry );

  /**
  Get first contact valid geometry in the shovel rigid body for contact. The geometry must be enabled
  and also not a sensor.
  \param shovelBody - The rigid body of the Shovel that will be checked.
  \return the first valid geometry for aggregate contacts available in the shovel body. If no one exists, return nullptr.
  */
  AGXTERRAIN_EXPORT agxCollide::Geometry* getFirstContactValidGeometry( const agx::RigidBody* shovelBody );

  /**
  Find the agx::Material on the first contact valid contact geometry in the specified shovel.
  \param shovel - the specified shovel.
  \return the agx::Material found on the first contact valid geometry in the specified shovel body.
          Returns nullptr if no contact valid geometry is found.
  */
  AGXTERRAIN_EXPORT agx::Material* findShovelMaterial( agxTerrain::Shovel* shovel );

  /**
  Utility method for estimating the contact depth of a shovel <-> aggregate contact from
  contact velocity and gravity.
  */
  agx::Real AGXTERRAIN_EXPORT estimateContactDepth(
    agx::ContactMaterial* contactMaterial,
    Shovel* shovel,
    agx::Real bodyMass,
    agx::Real contactVelocity,
    agx::Real contactGravity,
    agx::Real contactArea,
    agx::Real elasticRestLength,
    size_t numContacts );

  /**
  Utility method for calculating the relative velocity of two bodies in
  a contact normal direction.
  */
  agx::Real AGXTERRAIN_EXPORT calculateGv(
    const agx::Vec3& normal,
    const agx::Vec3& point,
    const agx::RigidBody* rb1,
    const agx::RigidBody* rb2 );

  /**
  Utility method for calculating the gravity vector given a point. Used in the
  shovel <-> aggregate contact model and in penetration resistance. This function is
  used to handle terrain gravity computation for different types gravity fields.
  */
  agx::Vec3 AGXTERRAIN_EXPORT calculateGravityVector(
    agxSDK::Simulation* simulation,
    const agx::Vec3& position );

  /**
  Computes the contact set between two aggregate in the cut top plane of shovel one
  */
  agxCollide::LocalGeometryContactVector AGXTERRAIN_EXPORT computeAggregatePairContacts( agxTerrain::Shovel* shovel1,
                                                                                         agxTerrain::Shovel* shovel2,
                                                                                         agx::ContactMaterial* contactMaterial = nullptr,
                                                                                         bool debugRenderContacts=false,
                                                                                         agx::Real size = 0.1);

}
