/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or
having been advised so by Algoryx Simulation AB for a time limited evaluation,
or having purchased a valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/Grid.h>
#include <agxTerrain/VoxelGridUtils.h>

namespace agxTerrain
{
  // Container for defining a height in terms of which voxel level and how filled the voxel is.
  struct GridElement
  {
    int z = 0;
    float occupancy = 0.0f;
  };

  /**
  Class implementing the BasicGrid interface.
  */
  class AGXTERRAIN_EXPORT ColumnHeightGrid : public agxTerrain::Grid
  {
  public:

    ColumnHeightGrid( const float defaultValue,
                      const size_t resolutionX,
                      const size_t resolutionY,
                      const agx::Real lowestMinimumHeight,
                      const agx::Real voxelSize,
                      const agx::RealVector& minimumHeights = agx::RealVector());

    ~ColumnHeightGrid();

    virtual agx::Vec3 getElementSize() const override;

    virtual size_t getMemoryUsage() const override;

    virtual bool empty() override;

    virtual void clearGrid() override;

    virtual void setFloatValue(const GridCoord& ijk, float value) override;

    virtual float getFloatValue(const GridCoord& ijk) const override;

    virtual bool isActiveVoxel(const GridCoord& /*ijk*/) const override;

    virtual size_t getNumActiveVoxels() const override;

    virtual agx::Real getTotal() const override;

    virtual std::tuple<float, float> getTotalAndLargest() const override;

    virtual GridCoordVector getActiveVoxelIndices() const override;

    virtual void sizeHint(size_t /*hint*/) override;

    virtual bool hasIterateSupport() const override;

    virtual void iterateVoxels(GridFunction func) const override;

    virtual agx::HashSet<agx::Vec3i> findGeometryVoxelIntersections(
      const agx::AffineMatrix4x4& gridTransform,
      const agxCollide::Geometry* geometry,
      agx::Real voxelSize,
      bool /*onlyOccupiedVoxels*/ = true,
      bool onlyMidPoint =false ) const override;

    agxCollide::BoundingAABB getBoundingBox() const override;

    virtual void setValuesBelowHeightField(
      const agxCollide::HeightField* heightField,
      agx::Real voxelSize,
      int lowestAllowedVoxelIndex,
      BasicGrid* compactionGrid) override;

    virtual agx::HashSet<agx::Vec3i> findConvexVoxelIntersections(
      const agx::AffineMatrix4x4& gridTransform,
      const agxCollide::Convex* convex,
      agx::Real voxelSize) override;

    virtual bool findMeshVoxelIntersections(
      const agx::AffineMatrix4x4& gridTransform,
      const agxCollide::Mesh* mesh,
      agx::Real voxelSize,
      agx::Vec3iVector& result) override;

    virtual agx::HashSet<agx::Vec3i> findVoxelsAboveTriangles(
      const agx::AffineMatrix4x4& gridTransform,
      const TriangleVector& triangles,
      agx::Real voxelSize,
      agx::Real margin) override;

    virtual ActiveZoneIntersectionData removeVolumeInActiveZone( const agx::AffineMatrix4x4& gridTransformation,
                                                                 const ActiveZoneShapes& activeZoneShapes,
                                                                 const MeshIntersectionTestData& innerShape,
                                                                 const agx::Plane& innerShapeBoundary,
                                                                 const agx::Bound3& activeZoneGeometryBound,
                                                                 const agx::Vec3iVector voxelsInPrimaryActiveZoneLastTimeStep,
                                                                 std::function<float(const agx::Vec3i&)> getCompression,
                                                                 std::function<bool(const agx::Vec2i&)> canRemoveIndex,
                                                                 agx::Real voxelSize,
                                                                 agx::Real minOccupancyRemoval,
                                                                 std::function<agx::Real(const agx::Vec2i&)> getMinAllowedHeight,
                                                                 bool useParticleFreeDeformers) override;

    virtual ActiveZoneIntersectionData removeVolumeInShape( const agx::AffineMatrix4x4& gridTransformation,
                                                            const ShapeIntersectionTestData& shape,
                                                            std::function<float( const agx::Vec3i& )> getCompression,
                                                            std::function<bool( const agx::Vec2i& )> canRemoveIndex,
                                                            std::function<bool( const agx::Vec3&, const agx::Vec3i& )> isWithinBoundry,
                                                            agx::Real voxelSize,
                                                            agx::Real minOccupancyRemoval,
                                                            std::function<agx::Real(const agx::Vec2i&)> getMinAllowedHeight ) override;

    virtual agx::Vec3iVector findVoxelsInLine( const agx::Line& gridLine ) const override;

    virtual void findVoxelSurfaceFromIndex( const agx::Vec3i& index,
                                            agx::Vec3i& result ) override;

    agx::Vec2i getResolution();

    agx::Real getLocalHeight(const agx::Vec2i terrainIndex);

    agx::Real getHeightFromVoxelZAndOccupancy(int voxelZ, float occupancy);

    int getSurfaceHeightIndex(const agx::Vec2i terrainIndex);

    std::pair<int, float> getVoxelZAndOccupancyFromLocalHeight(agx::Real height);

    void setGridColumnOccupancy(const agx::Vec2i terrainIndex, int voxelZ, float occupancy);

    void setOccupancyInSurfaceVoxel(const agx::Vec2i terrainIndex, float occupancy);

    bool isIndexOutsideBounds(agx::Vec2i index) const;

    /* Minimum height functionality */
    agx::Real getLowestMinimumHeight() const;

    void setLowestMinimumHeight(agx::Real minimumHeight);

    agx::Int getLowestAllowableVoxelIndexZ() const;

    agx::Real getMinimumHeight(agx::Vec2i terrainIndex) const;

    bool setMinimumHeight(agx::Vec2i terrainIndex, agx::Real minimumHeight);

    agx::Int getVoxelZFromHeight(agx::Real height) const;

    agx::Int getMinimumHeightIndexZ(agx::Vec2i terrainIndex) const;

    agx::RealVector getMinimumHeights() const;

    /*
    Set the minimum heights in the grid.
    \param minimumHeights - row major array of size (resX * resY).
    \note If getEnableMinimumHeights() == false, this method will print a warning and then enable minimum heights, and continue as usual.
    */
    void setMinimumHeights(const agx::RealVector& minimumHeights);

    bool getEnableMinimumHeights() const;

    /*
    Enable minimum heights grid. If disabled, getLowestMinimumHeight() will be used as the lowest point in all coordinates.
    */
    void setEnableMinimumHeights(bool enable);


    AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::ColumnHeightGrid);

  protected:

    inline agx::Vec3 getGridPositionFromVoxelIndex(const agx::Vec3i& voxelIndex) const
    {
      return VoxelGridUtils::getGridPositionFromVoxelIndex(voxelIndex, m_voxelSize);
    }

    inline agx::Vec3i getVoxelIndexFromGridPosition(agx::Vec3 position) const
    {
      return VoxelGridUtils::getVoxelIndexFromGridPosition(position, m_voxelSize);
    }

    ColumnHeightGrid();

    void allocateMassArray();

    void allocateMinHeights();

    void clearMinHeights();

  private:

    using GridData = agx::Vector<agx::Vector<GridElement>>;

    GridData    m_solidMassArray;
    float       m_default;
    size_t      m_resolutionX;
    size_t      m_resolutionY;
    agx::Real   m_voxelSize;

    // Minimum height grid data
    using HeightData = agx::Vector<agx::RealVector>;

    HeightData            m_minimumHeigths;
    bool                  m_enableMinHeights;
    agx::Real             m_lowestMinimumHeight;
    };
  }

