/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceHandles.h>

namespace agxSensor
{
  /**
  Base type for all surface, and transmission, materials used in raytracing.
  */
  class AGXSENSOR_EXPORT RtMaterial
  {
    public:
      /**
      Construct an invalid material.
      */
      RtMaterial();

      /**
      Construct given instance.
      */
      RtMaterial( RtMaterialInstance instance );

      /**
      \return true if valid, otherwise false
      */
      bool isValid() const;

      /**
      \param other - other material to compare
      \return true if the instance handle of this material equals the \p other instance handle
      */
      bool operator== ( const RtMaterial& other ) const;
      bool operator!= ( const RtMaterial& other ) const;

      /**
      \return handle of this material in the raytrace environment
      */
      RtMaterialHandleRef getHandle() const;

      /**
      \return the type of this material
      */
      RtMaterialHandle::Type getType() const;

      /**
      \return raytrace material instance
      */
      operator RtMaterialInstance() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    protected:
      RtMaterial( RtMaterialHandle::Type type );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      RtMaterialInstance m_instance;
  };

  inline RtMaterial::RtMaterial()
    : m_instance{}
  {
  }

  inline RtMaterial::RtMaterial( RtMaterialHandle::Type type )
    : m_instance{ RtMaterialInstance::create( type ) }
  {
  }

  inline RtMaterial::RtMaterial( RtMaterialInstance instance )
    : m_instance{ instance }
  {
  }

  inline bool RtMaterial::isValid() const
  {
    return m_instance.handle != nullptr;
  }

  inline bool RtMaterial::operator== ( const RtMaterial& other ) const
  {
    return m_instance.handle == other.m_instance.handle;
  }

  inline bool RtMaterial::operator!= ( const RtMaterial& other ) const
  {
    return !( (*this) == other );
  }

  inline RtMaterialHandleRef RtMaterial::getHandle() const
  {
    return m_instance.handle;
  }

  inline RtMaterialHandle::Type RtMaterial::getType() const
  {
    return m_instance.type;
  }

  inline RtMaterial::operator RtMaterialInstance() const
  {
    return m_instance;
  }
}
