/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>
#include <agxSensor/TriaxialCrossSensitivity.h>
#include <agxSensor/TriaxialRange.h>
#include <agxSensor/TriaxialSignalSystemNode.h>

#include <agxStream/Serializable.h>

#include <agx/Referenced.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( GyroscopeModel );

  /**
  Base gyroscope model describing the fundamental parameters, such as measurement range and
  zero bias, of a gyroscope.
  \note The configuration in this model is shared and synchronized across all instances.
  */
  class AGXSENSOR_EXPORT GyroscopeModel : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Creates a gyroscope model outputting ideal measurement data (ground truth values).
      \return new ideal gyroscope model
      */
      static GyroscopeModel* makeIdealModel();

    public:
      /**
      Constructs a gyroscope model with the specified parameters.
      \param range - detectable range of rotational rate, in radian/s
      \param crossSensitivity - cross axis sensitivity
      \param zeroRateBias - bias rotational rate reported in each axis under conditions without
                            externally applied rotational rate, in radian/s
      \param outputModifiers - list of modifiers to apply to the output of the gyroscope
      */
      GyroscopeModel( TriaxialRange range = agx::RangeReal(
                        std::numeric_limits<agx::Real>::lowest(),
                        std::numeric_limits<agx::Real>::max() ),
                      TriaxialCrossSensitivity crossSensitivity = 0.01,
                      agx::Vec3 zeroRateBias = agx::Vec3( 3.0 ),
                      ITriaxialSignalSystemNodeRefVector outputModifiers = ITriaxialSignalSystemNodeRefVector() );

      /**
      Set the detectable range for the gyroscope.
      \param range - detectable range of rotational rate, in radian/s
      */
      void setRange( const TriaxialRange& range );

      /**
      \return the range of rotational rate, in radian/s, that the gyroscope can detect.
      */
      const TriaxialRange& getRange() const;

      /**
      Set the gyroscope cross-axis sensitivity.
      \param crossSensitivity - cross axis sensitivity
      */
      void setCrossAxisSensitivity( const TriaxialCrossSensitivity& crossSensitivity );

      /**
      \return the cross sensitivity, describing the effects that rotational rate around one axis has
              on every other axis.
      */
      const TriaxialCrossSensitivity& getCrossAxisSensitivity() const;

      /**
      Set the bias in rotational rate at zero externally applied rotation.
      \param zeroRateBias - bias rotational rate reported in each axis under conditions without
                            externally applied rotational rate, in radian/s
      */
      void setZeroRateBias( const agx::Vec3& zeroRateBias );

      /**
      \return the bias rotational rate, in radian/s, reported in each axis at under conditions
              without externally applied rotation.
      */
      const agx::Vec3& getZeroRateBias() const;

      /**
      \return the list of output modifiers to apply to the output
      */
      const ITriaxialSignalSystemNodeRefVector& getOutputModifiers() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~GyroscopeModel() = default;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::GyroscopeModel );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      TriaxialRange m_range;
      TriaxialCrossSensitivity m_crossAxisSensitivity;
      agx::Vec3 m_zeroRateBias;
      ITriaxialSignalSystemNodeRefVector m_outputModifiers;
  };
}
