/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxOSG/export.h>
#include <agxOSG/GuiEventListener.h>
#include <agx/HashVector.h>
#include <agx/Constraint.h>
#include <agxUtil/Statistic.h>
#include <agxRender/Color.h>

namespace agxOSG
{
  /**
    Creates an arrow visualizing force,
    either contact force (for added bodies),
    or motor force for added Constraint1DOF (Hinge,Prismatic,..)

    The arrow is positioned on an local offset rel. the body associated.
    For constraints it is the first body of the first constraint.

    The size of the arrow is possible to scale, and the looks are tuned
    with cone length and arrow radius.
    */
  class AGXOSG_EXPORT ForceArrowRenderer : public agxSDK::GuiEventListener
  {
  public:
    
    /**
    Specifies which secondary constraints that should be included in the statistics.
    */
    enum ConstraintForceStatistics
    {
      MOTOR1D = 0x1,
      RANGE1D = 0x2,
      LOCK1D = 0x4,
      ALL = MOTOR1D | RANGE1D | LOCK1D,
      DEFAULT = MOTOR1D
    };

    /// Constructor
    ForceArrowRenderer();

    /**
    Add body for contact force visualization, normal and friction force are visualized with one arrow each.
    \param body - The rigid body that should be added.
    \param localOffsetRelBody - the starting point of the arrow relative the rigid body.
    \param color - Color of the arrow.
    \return true if body is successfully added.
    */
    bool add(agx::RigidBody* body, const agx::Vec3& localOffsetRelBody, const agxRender::Color& color = agxRender::Color::Red());

    /**
    Remove a body from the contact force arrow visualization.
    \param body - The rigidbody that should be removed
    \return true if body was found and removed.
    */
    agx::Bool remove(agx::RigidBody* body);


    /*
    Add a 1DOF constraint. The sum of the motor forces are visualized with one arrow.
    \param constraint - The constraint that should be added.
    \param localOffsetRelBody1 - the start point of the arrow relative the first rigid body of the first constraint.
    \param color - Color of the arrow.
    \param statisticsMask - Selects which forces that should be part of the calculation of the length of the force vector.
    \return true if constraint is successfully added.
    */
    bool add(agx::Constraint1DOF* constraint, const agx::Vec3& localOffsetRelBody1, const agxRender::Color& color = agxRender::Color::Red(), ConstraintForceStatistics statisticsMask = DEFAULT);

    /**
    Remove a constraint from the constraint force arrow visualization.
    \param constraint - The constraint that should be removed.
    */
    agx::Bool remove(agx::Constraint1DOF* constraint);

    /**
    Set the color of the force vector for the specified body
    \param body - the body that should be adressed.
    \param color - new color
    \return true if body exists.
    */ 
    agx::Bool setColor(agx::RigidBody* body, const agxRender::Color& color);

    /**
    Set the color of the force vector for the specified constraint
    \param constraint - the constraint that should be adressed.
    \param color - new color
    \return true if constraint exists.
    */
    agx::Bool setColor(agx::Constraint1DOF* constraint, const agxRender::Color& color);

    /**
    A smooth factor for a filter, smoothing the force arrow size.
    \param smoothFactor - factor to be used with a agxUtil::ExponentialMovingAverageStatistic
    */
    void setSmoothFactor(agx::Real smoothFactor);

    /**
    \return the current smooth factor
    */
    agx::Real getSmoothFactor() const;

    /**
    Set the unit height of the cone part of the arrow
    \param height - cone length
    */
    void setConeHeight(agx::Real height);

    /// \return the unit height of the cone part of the arrow
    agx::Real getConeHeight() const;
    
    /**
    Set the unit radius of the arrow.
    \param radius - arrow radius
    */
    void setArrowRadius(agx::Real radius);

    /**
    \return the unit radius of the arrow.
    */
    agx::Real getArrowRadius() const;

    /**
    Scale the size of the arrow (scale 1 would result in 1 Newton giving length of 1 l.u. )
    \param scale - a scale factor from Newton to length units (l.u.).
    */
    void setScale(agx::Real scale);

    /**
    \return the scale of the arrow.
    */
    agx::Real getScale() const;


  protected:
    virtual void update(float x, float y);
    virtual ~ForceArrowRenderer();

    void drawArrow(const agx::Vec3& start, const agx::Vec3& force, const agxRender::Color& color);

    agx::Bool m_enabled;
    agx::Real m_smoothFactor;
    agx::Real m_coneHeight;
    agx::Real m_arrowRadius;
    agx::Real m_scale;

#ifndef SWIG
    /**
    Contact force directions
    */
    enum ContactForceStatistics
    {
      NORMAL_X = 0,
      NORMAL_Y,
      NORMAL_Z,
      FRICTION_X,
      FRICTION_Y,
      FRICTION_Z,
      NUM_FORCES
    };

    template<typename T>
    struct Statistic : public agx::Referenced, public T
    {
      using T::T;
    };

    typedef Statistic<agxUtil::ExponentialMovingAverageStatistic> RendererStatistic;

    class ConstraintData : public agx::Referenced {
    public:
      ConstraintData(
        agx::Constraint1DOF* c, 
        RendererStatistic* s, 
        const agx::Vec3& r, 
        const agxRender::Color& col, 
        ConstraintForceStatistics mask) : constraint(c), 
                                          statistic(s), 
                                          relative(r), 
                                          statisticsMask(mask),
                                          color(col) {}
      

      agx::Constraint1DOFObserver constraint;
      agx::ref_ptr<RendererStatistic> statistic;
      agx::Vec3 relative;
      ConstraintForceStatistics statisticsMask;
      agxRender::Color color;
    };
    
    class BodyData : public agx::Referenced {
    public:
      BodyData(agx::RigidBody* b, 
               agx::Real smoothFactor, 
               const agx::Vec3& r, 
               const agxRender::Color& c) : body(b), relative(r), color(c)
      {
        for (int i = 0; i < NUM_FORCES; i++)
          statistics.push_back(new RendererStatistic(smoothFactor));
      }

      agx::RigidBodyObserver body;
      agx::Vector<agx::ref_ptr<RendererStatistic>> statistics;
      agx::Vec3 relative;
      agxRender::Color color;
    };

    typedef agx::HashVector<agx::Constraint1DOF*, agx::ref_ptr< ConstraintData>> ConstraintHashVector;
    typedef agx::HashVector<agx::RigidBody*, agx::ref_ptr<BodyData>> RigidBodyHashVector;
    
    ConstraintHashVector m_constraints;
    RigidBodyHashVector m_bodies;
#endif
  };

}