/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <memory>

#include <agx/String.h>

#include <agxSDK/LinkedStructureComponent.h>

#include <agxCable/Cable.h>
#include <agxCable/export.h>

namespace agxCable
{
  AGX_DECLARE_POINTER_TYPES(CableTunnelingGuard);

  /**
  A linked structure componenent which uses additional sensor geometries placed around each segment to register
  addtiional context to the solver as to reduce the preseence of tunneling between cables
  */
  class AGXCABLE_EXPORT CableTunnelingGuard : public agxSDK::LinkedStructureComponent
  {
  public:
    AGXSTREAM_DECLARE_SERIALIZABLE(agxCable::CableTunnelingGuard);
    /**
    Get a shared reference to a CableTunnelingGuard on \p cable
    \param cable The cable to get the component from
    \returns A reference to the componenet or nullptr if no CableTunnelingGuard is present
    */
    static CableTunnelingGuard* getCableTunnelingGuard(agxCable::Cable* cable);

    /**
    Constructs an instance with hull geometries \p hullScale times thicker than the cable itself.
    \param hullScale the initial hull scale
     */
    CableTunnelingGuard(double hullScale);

    /**
    Sets the hull scale to \p scale
    \param scale The new hull scale
    */
    void setHullScale(double scale);

    /**
    Gets the current hull scale
    \returns The current hull scale
    */
    agx::Real getHullScale() const;

    /**
    Gets the cable to which the component is added
    \returns A cable object or nullptr if the component is not added to a cable
    */
    agxCable::Cable* getCable() const;

    /**
    Get the threshold angle (in radians) towards the cap of each hull geometry from the contacts normal at which the
    contact is discarded. Contacts at the caps of the hulls are either: two ends of a cable approaching each other
    (which rarely result in tunneling), or a contact falsely generated with a neighbouring hull when the other cable is
    moving is within a hull. To avoid these contacts, they are discarded if the normal is directed above the threshold
    angle towards the length of the hull. \note The caps are naturally exposed when segments are bent at vary sharp
    angles. This parameter may need to be increased in such extreme cases to avoid culling too many contacts. \returns
    The angle threshold
    */
    agx::Real getAngleThreshold() const;

    /**
    Set the angle threshold to \p radians
    \param radians The new value
    */
    void setAngleThreshold(agx::Real radians);

    /**
    Gets the leniency of the component; a measure of how deep the next estimated contact depth needs to be to be added
    as an additional contact. \note The formula for calculating the required depth is: collidingCable->getRadius() * 0.5
    * (1 - leniency). This means that values 0-1 needs the estimated depth to be inside the colliding cable and that
    values > 1 do not require this. \returns The leniency of the component
    */
    agx::Real getLeniency() const;

    /**
    Sets the leniency
    \param value The new leniency
    */
    void setLeniency(agx::Real value);

    /**
    Gets the number of steps for which contacts with a geomtry are added regardless of other criteria after first
    contact. \returns The number of debounce steps
    */
    agx::UInt32 getDebounceSteps() const;

    /**
    Sets the number of debounce steps
    \param value The new value
    */
    void setDebounceSteps(agx::UInt32 value);

    /**
    Gets the bool deciding if the component should always add hull contacts to the solver, disregarding other
    optimisations and considerations. \returns True if the component will always add contacts, false otherwise
    */
    agx::Bool getAlwaysAdd() const;

    /**
    Sets the component to always add hull contacts no matter what if \p value is true and to not do so if \p value is
    false. \p value The new value.
    */
    void setAlwaysAdd(agx::Bool value);

    /**
    Gets the bool deciding if the component should consider contacts with it's own segments, preventing self-tunneling
    \returns True if the component will predict contacts with it's own segments
    */
    agx::Bool getEnableSelfInteraction() const;

    /**
    Sets the component to consider self-collisions if \p value is true or not to, if \p value is false.
    \p value The new value.
    */
    void setEnableSelfInteraction(agx::Bool value);

    /**
    Gets the number of contacts added to the solver since the components creation
    \returns The number of added contacts
    */
    agx::Size getNumAddedContacts() const;

    /**
    Gets the number of contacts added to the solver since the components creation
    \returns The number of encountered contacts
    */
    agx::Size getNumEncounteredContacts() const;

    /**
    Sets the enabled state of the component. When disabled, the component will have no effect on the behaviour of the
    cable. \p value The new value.
    */
    void setEnabled(agx::Bool value);

    /**
    Returns true if the component is enabled.
    \returns The enabled state of the component
    */
    agx::Bool getEnabled() const;

  protected:
    CableTunnelingGuard();

    virtual void onAddNotification(agxSDK::Simulation* simulation) override;
    virtual void onAddNotification(agxSDK::LinkedStructure* linkedStructure) override;
    virtual void onRemoveNotification(agxSDK::LinkedStructure* linkedStructure) override;
    virtual void onRemoveNotification(agxSDK::Simulation* simulation) override;
    virtual void onPreStep() override;
    virtual void onLastStep() override;

  private:
    friend class HullContactListener;

    CableObserver m_cable;

    agxSDK::ContactEventListenerRef m_listener;
    agxCollide::GeometryRefVector   m_addedGeometries;

    agx::Real   m_hullScale;
    agx::Size   m_numAddedContacts;
    agx::Size   m_numEncounteredContacts;
    agx::Real   m_cosAngleThreshold;
    agx::Real   m_leniency;
    agx::UInt32 m_debounceSteps;
    agx::Bool   m_alwaysAdd;
    agx::Bool   m_selfInteractionEnabled;
    agx::Bool   m_enabled;

    void enable();
    void disable();

    void createHulls();
    void destroyHulls(agxSDK::LinkedStructure* linkedStructure);
    void updateHulls(agx::Real oldHullScale = -agx::Infinity);
  };
} // namespace agxCable
