/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#ifndef AGX_CONSTRAINTANGLE_H
#define AGX_CONSTRAINTANGLE_H

#include <agx/Vec3.h>

#include <agxStream/Serializable.h>

namespace agx
{
  class AttachmentPair;

  /**
  Constraint angle class.
  */
  class AGXPHYSICS_EXPORT Angle : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Axis this constraint angle is defined about or along.
      */
      enum Axis
      {
        U, /**< Constraint x axis. */
        V, /**< Constraint y axis. */
        N  /**< Constraint z axis. */
      };

      /**
      Type, i.e., defines \p Axis as 'along' or 'about'.
      */
      enum Type
      {
        TRANSLATIONAL, /**< \p Axis is along U, V or N. */
        ROTATIONAL     /**< \p Axis is about U, V or N. */
      };

    public:
      /**
      Transform method to update the value of this angle.
      \param ap - attachment pair this angle is part of
      */
      virtual void transform( const agx::AttachmentPair& ap );

      /**
      Post solvers, velocity dependent variables should be updated.
      \param ap - attachment pair this angle is part of
      \param dt - time step size
      */
      virtual void onPostSolve( const agx::AttachmentPair& ap, agx::Real dt );

      /**
      \return the axis this constraint angle is defined about or along
      */
      agx::Angle::Axis getAxis() const;

      /**
      \return the type of this constraint angle (i.e., if \p Axis is 'along' or 'about')
      */
      agx::Angle::Type getType() const;

      /**
      \return const reference to value (so it may be used by elementary constraints that works on this angle)
      */
      agx::Real getValue() const;

      /**
      \param ap    - attachment pair
      \param index - which attachment (0,1)
      \return the world direction of the axis given the attachment pair
      */
      virtual const agx::Vec3& getDir( const agx::AttachmentPair& ap, agx::UInt index ) const;

      /**
      Store given stream.
      */
      virtual void store( agxStream::OutputArchive& out ) const override;

      /**
      Restore given stream.
      */
      virtual void restore( agxStream::InputArchive& in ) override;

      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agx::Angle );

      /**
      Store structural independent data to stream.
      */
      virtual void storeLightData( agxStream::StorageStream& str ) const override;

      /**
      Restore structural independent data from stream.
      */
      virtual void restoreLightData( agxStream::StorageStream& str ) override;

    protected:
      /**
      Construct given axis and initial angle.
      */
      Angle( agx::Angle::Axis axis, agx::Angle::Type type, agx::Real initialAngle = agx::Real( 0 ) );

      /**
      Reference counted object, protected destructor.
      */
      virtual ~Angle();

    protected:
      Axis      m_axis;
      Type      m_type;
      agx::Real m_value;
  };

  typedef agx::ref_ptr< Angle> AngleRef;
  typedef agx::Vector<AngleRef> AngleContainer;

  /**
  Implementation of constraint angle ABOUT an axis.
  */
  class AGXPHYSICS_EXPORT RotationalAngle : public agx::Angle
  {
    public:
      /**
      Utility cast method.
      */
      static agx::RotationalAngle* safeCast( const agx::Angle* angle );

    public:
      /**
      Construct given axis and initial angle. Note that the given \p initialAngle
      has to match the angle of the frames in the attachment pair this angle
      belongs to.
      \param axis - axis this angle is defined about
      \param initialAngle - initial angle
      */
      RotationalAngle( agx::Angle::Axis axis, agx::Real initialAngle = agx::Real( 0 ) );

      /**
      Update value given constraint pair.
      */
      virtual void transform( const agx::AttachmentPair& ap ) override;

      /**
      Updates winding number given angular speed about our axis.
      \param ap - attachment pair of this angle
      \param dt - time step size
      */
      virtual void onPostSolve( const agx::AttachmentPair& ap, agx::Real dt ) override;

      /**
      \return the winding number
      */
      agx::Int getWindingNumber() const;

      /**
      Use with caution since each values of \p windingNumber represent plus or minus
      360 degrees of the joint angle (e.g, affecting locks and ranges).
      \param windingNumber - new windingNumber
      */
      void setWindingNumber( agx::Int windingNumber );

      /**
      \return the previous angle (-pi, +pi) (winding number NOT included)
      */
      agx::Real getLastAngle() const;

      /**
      Set the last angle bewteen (-pi, +pi) (winding number NOT included)
      \param lastAngle - will be clamped in range (-pi, +pi) if outside
      */
      void setLastAngle(agx::Real lastAngle);

      AGXSTREAM_DECLARE_SERIALIZABLE(agx::RotationalAngle);

      /**
      Store structural independent data to stream.
      */
      virtual void storeLightData( agxStream::StorageStream& str ) const override;

      /**
      Restore structural independent data from stream.
      */
      virtual void restoreLightData( agxStream::StorageStream& str ) override;

    protected:
      /**
      Default constructor used during restore.
      */
      RotationalAngle();

    protected:
      /**
      Reference counted object, protected destructor.
      */
      virtual ~RotationalAngle();

      /**
      Integrate/step parameters given a delta angle.
      \param deltaAngle - delta angle (any sign, any value)
      */
      virtual void stepBy( agx::Real deltaAngle );

    protected:
      agx::Int  m_windingNumber;
      agx::Real m_lastAngle;
  };

  typedef agx::ref_ptr<RotationalAngle> RotationalAngleRef;

  /**
  Implementation of constraint angle ALONG an axis.
  */
  class AGXPHYSICS_EXPORT SeparationAngle : public agx::Angle
  {
    public:
      /**
      Utility cast method.
      */
      static agx::SeparationAngle* safeCast( const agx::Angle* angle );

    public:
      /**
      Construct given axis and initial angle.
      */
      SeparationAngle( agx::Angle::Axis axis, agx::Real initialAngle = agx::Real( 0 ) );

      /**
      Update value given constraint pair.
      */
      virtual void transform( const agx::AttachmentPair& ap ) override;

      AGXSTREAM_DECLARE_SERIALIZABLE(agx::SeparationAngle);

    protected:
      /**
      Default constructor used during restore.
      */
      SeparationAngle();

      /**
      Reference counted object, protected destructor.
      */
      virtual ~SeparationAngle();
  };

  typedef agx::ref_ptr<SeparationAngle> SeparationAngleRef;



  /**
  Implementation of angle for the cone limit. The Angle is defined around the n-axis, but only the friction controller
  for rotation on the axis actually uses that axis. In the case of the friction along the limit, another direction is
  calculated, so that it is along the limit. For the ConeLimit itself, the direction is not used at all, the angle is 
  however used to save the limits so that the FrictionController along the limit also can use it.
  */
  class AGXPHYSICS_EXPORT ConeLimitAngle : public Angle
  {
  public:
    /**
    Constructor for cone limit angle
    \param useRotAngle - Set to true to remove the special direction along the cone limit
    */
    ConeLimitAngle(bool useRotAngle);

    /**
    Default constructor
    */
    ConeLimitAngle();

    /**
    Transform method to update the values of this Angle.
    */
    virtual void transform(const AttachmentPair& ap) override;

    /**
    Get the direction along the cone limit or of the axis given the attachment pair, depending on the
    case the Angle is used in.
    \param ap    - attachment pair
    \param index - which attachment (0,1)
    \return Vec3 describing the direction, in the world
    */
    virtual const Vec3& getDir(const AttachmentPair& ap, agx::UInt index) const override;

    /**
    Set the ConeLimit range, in radians. Note that if this Angle is not the same as in the ConeLimit,
    this will not set the limit.
    \param limit - Vec2 containing the two limit angles, in radians
    */
    void setConeLimitRange(Vec2 limit);

    /**
    Get the ConeLimit range, in radians. Note that if this Angle is not the same as in the ConeLimit,
    this will not return the correct limit.
    \return Vec2 containing the two limit angles, in radians
    */
    Vec2 getConeLimitRange() const;

    AGXSTREAM_DECLARE_SERIALIZABLE(ConeLimitAngle);

  protected:
    bool m_useRotAngle;

  private:
    Vec2 m_currentLimit;
    Vec3 m_dir;
  };

  typedef ref_ptr<ConeLimitAngle> ConeLimitAngleRef;

  inline Real Angle::getValue() const
  {
    return m_value;
  }
} //namespace agx

#endif
