#pragma once

#include <string>
#include <memory>
#include <fmt/format.h>

namespace openplx {

    enum class LogLevel {
        Error,
        Warning,
        Info,
        Debug
    };

    class Logger {
        public:
            virtual ~Logger() = default;
            virtual void log(std::string message, LogLevel log_level) = 0;

            void enableDebugLogs() {
                m_enable_debug = true;
            }

            void disableDebugLogs() {
                m_enable_debug = false;
            }

            bool isDebugLogsEnabled() {
                return m_enable_debug;
            }

        private:
            bool m_enable_debug = false;
    };

    template <class... Args>
    void plxlog_base(
        std::shared_ptr<openplx::Logger> logger,
        const char* file, int line,
        const char* log_level_str,
        LogLevel log_level,
        fmt::format_string<Args...> fmtstr, Args&&... args)
    {
        if (logger == nullptr) return;
        if (!logger->isDebugLogsEnabled() && log_level == LogLevel::Debug) {
            return;
        }
        auto payload = fmt::format(fmtstr, std::forward<Args>(args)...);
        logger->log(fmt::format("[{}] {}:{}: {}", log_level_str, file, line, payload),
                    log_level);
    }
}

#define PLXLOG_INFO(logger, ...) \
    do { \
        openplx::plxlog_base((logger), __FILE__, __LINE__, "Info",  openplx::LogLevel::Info, __VA_ARGS__); \
    } while (0)
#define PLXLOG_WARN(logger, ...) \
    do { \
        openplx::plxlog_base((logger), __FILE__, __LINE__, "Warning",  openplx::LogLevel::Warning, __VA_ARGS__); \
    } while (0)
#define PLXLOG_ERROR(logger, ...) \
    do { \
        openplx::plxlog_base((logger), __FILE__, __LINE__, "Error", openplx::LogLevel::Error, __VA_ARGS__); \
    } while (0)
#define PLXLOG_DEBUG(logger, ...) \
    do { \
        openplx::plxlog_base((logger), __FILE__, __LINE__, "Debug", openplx::LogLevel::Debug, __VA_ARGS__); \
    } while (0)
