
#pragma once

#include <string>
#include <vector>
#include <memory>
#include <unordered_map>
#include <openplx/Token.h>
#include <openplx/ErrorReporter.h>
#include <openplx/Document.h>
#include <openplx/BundleLookup.h>

namespace std {
    template<>
    struct hash<openplx::Token> {
        inline size_t operator()(openplx::Token token) const {
            std::size_t hashValue = 0;
            hashValue ^= std::hash<openplx::TokenType>()(token.type);
            hashValue ^= std::hash<openplx::line_t>()(token.line);
            hashValue ^= std::hash<openplx::col_t>()(token.column);
            hashValue ^= std::hash<std::string>()(token.lexeme);
            return hashValue;
        }
    };
}

namespace openplx {

    struct TokenHighlightData {
        std::string classifier;
        TypePtr type;
    };

    class Highlighter : public NodeVisitor
    {
        public:
            Highlighter(const std::string& link_prefix);

            std::string highlightForJekyll(const std::string& source, const DocPtr& document, bool ligatures = false);

            void visitDocument(const DocPtr &document) override;
            void visitModelDeclaration(const ModelDeclPtr &model_declaration) override;
            void visitVarAssignment(const VarAssignPtr &var_assignment) override;
            void visitMethodDeclaration(const MethodDeclPtr &method_declaration) override;
            void visitConstant(const ConstantPtr &constant) override;
            void visitCall(const CallPtr &call) override;
            void visitMemberAccess(const MemberAccessPtr &member_access) override;
            void visitArray(const ArrayPtr &array) override;
            void visitUnary(const UnaryPtr &unary) override;
            void visitBinaryOp(const BinaryOpPtr &binary_op) override;
            void visitPrimitiveType(const PrimitiveTypePtr &primitive_type) override;
            void visitArrayType(const ArrayTypePtr &array_type) override;
            void visitParameter(const ParamPtr &parameter) override;
            void visitIndexing(const IndexingPtr &indexing) override;
            void visitOperatorOverload(const OpOverloadPtr &op_overload) override;
            void visitAnnotation(const AnnotationPtr &annotation) override;

        private:
            void classifyToken(const Token& token, std::string classifier = "");
            void classifyTypeSegments(const std::vector<Token>& tokens, const TypePtr& type);
            void linkToken(const Token& token, const TypePtr& type);

            std::string m_link_prefix;
            std::unordered_map<Token, TokenHighlightData> m_token_data_map;
    };
}
