/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>

#include <agx/Referenced.h>
#include <agx/BitState.h>

#include <agxStream/Serializable.h>

#include <queue>
#include <stack>

namespace agxSensor
{
  class Environment;

  AGX_DECLARE_POINTER_TYPES( SystemNode );
  AGX_DECLARE_VECTOR_TYPES( SystemNode );

  /**
  System node is a separate step/operation in an execution tree
  representing a sensor (or such) in a sensor simulation environment.
  */
  class AGXSENSOR_EXPORT SystemNode : public agx::Referenced, public agxStream::Serializable
  {
    public:
      struct CallbackData
      {
        agx::Real dt = 1.0 / 60.0;
      };

    public:
      /**
      Assign name to this node.
      \param name - new name of this node
      */
      void setName( const agx::Name& name );

      /**
      \return the name of this node
      */
      const agx::Name& getName() const;

      /**
      Enable/disable this node. Default enabled.
      \param enable - true (default) to enable, false to disable this node
      */
      void setEnable( bool enable );

      /**
      \return true if this node is enabled, false if disabled
      */
      bool getEnable() const;

      /**
      Add unique child to this node.
      \param child - child to add
      \return true if added successfully, false if nullptr or already a child 
      */
      bool addChild( SystemNode* child );

      /**
      Append child to vector of children, silently. This method will just
      check so that \p child hasn't already been added and set its parent
      (and push_back in vector) if not.
      \param child - child to add
      \return true if added, false if nullptr or already added
      */
      bool appendChild( SystemNode* child );

      /**
      Remove child from this node.
      \param child - child to remove
      \return true if removed successfully, false if nullptr or not child to this node
      */
      bool removeChild( SystemNode* child );

      /**
      Detach/erase child from vector of children, silently. This method will just
      check so that \p child is a child and has this node as parent.
      \param child - child to erase
      \return true if erased, false if nullptr or not child to this node
      */
      bool detachChild( SystemNode* child );

      /**
      Inserts new parent to this node. The previous parent to this node
      will get \p parent as child.
      \param parent - new parent to this node
      \return true if successful, false if initialization of RtNode of \p parent failed
      */
      virtual bool insertParent( SystemNode& parent );

      /**
      Erase this node from the tree and connect its children to the current
      parent of this node. Fails if this node doesn't have any children.
      \note This node may be destructed if no references are held to this node.
      \return true if successfully erased 
      */
      virtual bool eraseAndConnect();

      /**
      \param child - possible child node
      \param recurse - true to continue search to the leafs, false to only
                       search this parents children
      \return the if \p child is child to this node, otherwise false
      */
      bool hasChild( const SystemNode* child, bool recurse = true ) const;

      /**
      \return the number of children of this node
      */
      size_t getNumChildren() const;

      /**
      \param index - index of child
      \return child with index \p index
      */
      SystemNode* getChild( size_t index ) const;

      /**
      Visit the tree from, including, this node until \p callback returns
      true - using breadth first traversal.
      \tparam VisitorT - implicit visitor type bool(SystemNode&) or similar
      \param visitor - node visitor returning true if match, return false to
                       continue to the next node
      \return node when \p visitor returns true, nullptr if no match
      */
      template<typename VisitorT>
      SystemNode* visitBf( VisitorT visitor ) const;

      /**
      Visit the tree from the children of this node until \p callback returns
      true - using breadth first traversal.
      \tparam VisitorT - implicit visitor type bool(SystemNode&) or similar
      \param visitor - node visitor returning true if match, return false to
                       continue to the next node
      \return node when \p visitor returns true, nullptr if no match
      */
      template<typename VisitorT>
      SystemNode* visitChildrenBf( VisitorT visitor ) const;

      /**
      Visit the tree from, including, this node until \p callback returns
      true - using depth first traversal.
      \tparam VisitorT - implicit visitor type bool(SystemNode&) or similar
      \param visitor - node visitor returning true if match, return false to
                       continue to the next node
      \return node when \p visitor returns true, nullptr if no match
      */
      template<typename VisitorT>
      SystemNode* visitDf( VisitorT visitor ) const;

      /**
      Visit the tree from the children of this node until \p callback returns
      true - using depth first traversal.
      \tparam VisitorT - implicit visitor type bool(SystemNode&) or similar
      \param visitor - node visitor returning true if match, return false to
                       continue to the next node
      \return node when \p visitor returns true, nullptr if no match
      */
      template<typename VisitorT>
      SystemNode* visitChildrenDf( VisitorT visitor ) const;

      /**
      Searches children for a node of given type. Note that it's only the
      children of which this node is parent that is searched.
      \return child with type T if found, otherwise nullptr
      \sa findChild
      */
      template<typename T>
      T* getChild() const;

      /**
      Find child given predicate \p pred. If \p recurse is true, the search
      continues to the leafs.
      \param pred - predicate bool( const SystemNode& child ) returning true
                    if a child is a match, otherwise false
      \param recurse - true to continue search to the leafs, false to only
                       search this parents children
      \return child matching \p pred, otherwise nullptr for no match
      */
      template<typename PredicateT>
      SystemNode* findChild( PredicateT pred, bool recurse = true ) const;

      /**
      Find child of given type. If \p recurse is true, the search
      continues to the leafs.
      \param recurse - true to continue search to the leafs, false to only
                       search this parents children
      \return first child of given type, nullptr if not found
      */
      template<typename T>
      T* findChild( bool recurse = true ) const;

      /**
      \return parent of this node, nullptr if this is the root node
      */
      SystemNode* getParent() const;

      /**
      Find parent given predicate.
      \param pred - predicate bool( const SystemNode& parent ) returning true
                    if parent is a match, otherwise false
      \return parent matching \p pred, otherwise nullptr for no match
      */
      template<typename PredicateT>
      SystemNode* findParent( PredicateT pred ) const;

      /**
      \return first parent of given type, nullptr if not found
      */
      template<typename T>
      T* findParent() const;

      /**
      Visit all children with \p func as void( const SystemNode& child ).
      \param visitor - function to call for each child, void( SystemNode& child )
      \param recurse - true to continue the traverse to the leafs, false to only
                       traverse this parents children
      */
      template<typename VisitorT>
      void visitChildren( VisitorT visitor, bool recurse = true ) const;

      /**
      Visit all children with \p func as void( const T& child ).
      \param visitor - function to call for each child, void( T& child )
      \param recurse - true to continue the traverse to the leafs, false to only
                       traverse this parents children
      */
      template<typename T, typename VisitorT>
      void visitChildrenOfType( VisitorT visitor, bool recurse = true) const;

      /**
      \return the node with nullptr parent
      */
      SystemNode* getRoot() const;

      /**
      \return the sensor environment this node is part of
      */
      Environment* getEnvironment() const;

    public:
      /**
      Disallow direct copying of nodes.
      */
      SystemNode( const SystemNode& ) = delete;

      virtual ~SystemNode() = default;

      /**
      Called when this system node has been added to a sensor environment.
      \return true if successful and is ready to receive runtime calls
      */
      virtual bool addNotification();

      /**
      Called when this system node is being removed from a sensor environment.
      */
      virtual void removeNotification();

      /**
      Called when \p child has been added to this parent.
      \param child - child that has been added
      */
      virtual void onChildAdded( SystemNode& child );

      /**
      Called when \p child has been removed from this parent.
      \param child - child that has been removed
      */
      virtual void onChildRemoved( SystemNode& child );

      /**
      Called when an object has been added to the sensor environment.
      \param instance - object that was added to the sensor environment
      \return true if the object is consumed by the system, false if this
              system node doesn't handle/support this object.
      */
      virtual bool onAdd( agx::Referenced& instance );

      /**
      Called when an object has been removed from the sensor environment.
      \param instance - object that was removed from the sensor environment
      \return true if the object is consumed by the system, false if this
              system node doesn't handle/support this object.
      */
      virtual bool onRemove( agx::Referenced& instance );

      /**
      Synchronization from the main thread. Read data from objects/simulation
      should be done here before execute/stepping is made separate from the
      main thread.
      \param data - runtime step data
      */
      virtual void synchronize( const CallbackData& data );

      /**
      Execute this system, integrating it \p data.dt forward in time. This call 
      is made from a thread separate from the main/simulation thread so it's not
      valid to read data from the objects being simulated.
      \param data - runtime step data
      */
      virtual void execute( const CallbackData& data );
      
      /**
      Complete the execution of this system, finishing the integration forward in time. This call is
      made from a thread separate from the main/simulation thread so it's not valid to read data
      from the objects being simulated.
      */
      virtual void complete();

      /**
      If necessary assemble, and fetch results related to this system node after a step by
      \p data.dt has been completed. This call is made from the main thread, post-step, and thus
      allows the latest possible data from objects/simulation to be read and processed.
      \param data - runtime step data
      */
      virtual void result( const CallbackData& data );

      /**
      The sensor environment is being put down, clean everything.
      */
      virtual void cleanup() = 0;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      /**
      Write a string representation of the tree from this node and downwards to the given output stream.
      \param out - output stream to write this node's tree to
      */
      void printTree(std::ostream& out) const;

      /**
      Write a string representation of the tree from this node and downwards to the given output stream.
      \param out - output stream to write this node's tree to
      \param depth - node depth tracking/offset value
      */
      void printTree(std::ostream& out, size_t& depth) const;

      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::SystemNode );

      virtual void store( agxStream::OutputArchive& out ) const override;
      virtual void restore( agxStream::InputArchive& in ) override;

    protected:
      SystemNode();

    protected:
      SystemNode* m_parent;
      SystemNodeRefVector m_children;

      struct CallbackEvent
      {
        enum CallbackType
        {
          SYNC,
          EXECUTE,
          COMPLETE,
          RESULT,
          CLEANUP
        };

        CallbackType type;
        CallbackData data;
      };

      virtual void invoke( const CallbackEvent& callbackEvent );

    private:
      friend Environment;

      struct InternalState
      {
        enum Flag : agx::UInt32
        {
          ENABLED = 1 << 0,
        };
        using Flags = agx::BitState<Flag, std::underlying_type_t<Flag>>;
        Flags flags;

        InternalState()
          : flags( ENABLED )
        {
        }
      };

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      bool setEnvironment( Environment* environment );

    private:
      Environment* m_environment;
      agx::Name m_name;
      InternalState m_state;
  };

  template<typename VisitorT>
  SystemNode* SystemNode::visitBf( VisitorT visitor ) const
  {
    auto self = const_cast<SystemNode*>( this );
    if ( visitor( *self ) )
      return self;

    return this->visitChildrenBf( visitor );
  }

  template<typename VisitorT>
  SystemNode* SystemNode::visitChildrenBf( VisitorT visitor ) const
  {
    using SystemNodeQueue = std::queue<SystemNode*>;

    SystemNodeQueue queue;

    for ( auto& child : m_children )
      queue.push( child );

    while ( !queue.empty() ) {
      auto node = queue.front();
      queue.pop();

      if ( visitor( *node ) )
        return node;

      for ( auto& child : node->m_children )
        queue.push( child );
    }

    return nullptr;
  }

  template<typename VisitorT>
  SystemNode* SystemNode::visitDf( VisitorT visitor ) const
  {
    auto self = const_cast<SystemNode*>( this );
    if ( visitor( *self ) )
      return self;

    return this->visitChildrenDf( visitor );
  }

  template<typename VisitorT>
  SystemNode* SystemNode::visitChildrenDf( VisitorT visitor ) const
  {
    using SystemNodeStack = std::stack<SystemNode*>;

    SystemNodeStack stack;

    for ( auto& myChild : m_children ) {
      stack.push( myChild );

      while ( !stack.empty() ) {
        auto node = stack.top();
        stack.pop();

        if ( visitor( *node ) )
          return node;

        for ( auto it = node->m_children.rbegin(); it != node->m_children.rend(); ++it )
          stack.push( *it );
      }
    }

    return nullptr;
  }

  template<typename T>
  T* SystemNode::getChild() const
  {
    return this->template findChild<T>( false );
  }

  template<typename PredicateT>
  SystemNode* SystemNode::findChild( PredicateT pred, bool recurse /*= true*/ ) const
  {
    if ( recurse ) {
      return this->visitChildrenDf( [&pred]( const SystemNode& node )
                                    {
                                      return pred( node );
                                    } );
    }
    else {
      auto it = std::find_if( m_children.begin(),
                              m_children.end(),
                              [&pred]( const SystemNodeRef& child )
                              {
                                return child != nullptr && pred( *child.get() );
                              } );
      return it != m_children.end() ?
               (SystemNode*)it->get() :
               nullptr;
    }
  }

  template<typename T>
  T* SystemNode::findChild( bool recurse /*= true*/ ) const
  {
    auto child = this->findChild( []( const SystemNode& child )
                                  {
                                    return child.template asSafe<T>() != nullptr;
                                  },
                                  recurse );
    return child != nullptr ? child->template as<T>() : nullptr;
  }

  template<typename VisitorT>
  void SystemNode::visitChildren( VisitorT visitor, bool recurse /*= true*/ ) const
  {
    if ( recurse ) {
      this->visitChildrenBf( [&visitor]( SystemNode& child )
                             {
                               visitor( child );
                               return false;
                             } );
    }
    else {
      for ( auto& child : m_children )
        visitor( *child );
    }
  }

  template<typename PredicateT>
  SystemNode* SystemNode::findParent( PredicateT pred ) const
  {
    auto parent = const_cast<SystemNode*>( this );
    while ( ( parent = parent->getParent() ) != nullptr && !pred( *parent ) )
      ;

    return parent;
  }

  template<typename T>
  T* SystemNode::findParent() const
  {
    auto parent = this->findParent( []( const SystemNode& parent )
                                    {
                                      return parent.template asSafe<T>() != nullptr;
                                    } );
    return parent != nullptr ? parent->template asSafe<T>() : nullptr;
  }

  template<typename T, typename VisitorT>
  void SystemNode::visitChildrenOfType( VisitorT visitor, bool recurse /*= true*/ ) const
  {
    this->visitChildren( [&visitor]( SystemNode& child )
                         {
                           if ( auto childT = child.template asSafe<T>() )
                             visitor( *childT );
                         },
                         recurse );
  }
}
