/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceSurfaceMaterial.h>

namespace agxStream
{
  class OutputArchive;
  class InputArchive;
}

namespace agxSensor
{
  /**
  Opaque surface material reflecting the light according to the Lambertian model of diffuse 
  reflectance.

  This surface reflection model should suffice for most smooth diffuse surfaces.
  */
  class AGXSENSOR_EXPORT RtLambertianOpaqueMaterial : public RtSurfaceMaterial
  {
    public:
      static constexpr RtMaterialHandle::Type Type = RtMaterialHandle::Type::OPAQUE_LAMBERTIAN;

    public:
      /**
      Create a new lambertian opaque surface material in the
      raytrace environment. This material can be assigned any
      number of raytrace shape instances.
      \return new instance of a surface material
      */
      static RtLambertianOpaqueMaterial create();

      /**
      \return the surface material associated with the given shape
      */
      static RtLambertianOpaqueMaterial get( const agxCollide::Shape* shape );

      /**
      \return the surface material associated with the given geometry
      */
      static RtLambertianOpaqueMaterial get( const agxCollide::Geometry* geometry );

      /**
      \return the surface material associated with the given rigid body
      */
      static RtLambertianOpaqueMaterial get( const agx::RigidBody* rb );

      /**
      \return the surface material associated with the given linked structure (agxCable::Cable, agxModel::Beam, agxVehicle::Track, ...)
      */
      static RtLambertianOpaqueMaterial get( const agxSDK::LinkedStructure* linkedStructure );

      /**
      \return the surface material associated with the given wire
      */
      static RtLambertianOpaqueMaterial get( const agxWire::Wire* wire );

      /**
      \return the surface material associated with the given terrain
      */
      static RtLambertianOpaqueMaterial get( const agxTerrain::Terrain* terrain );

      /**
      \return the surface material associated with the given terrain pager
      */
      static RtLambertianOpaqueMaterial get( const agxTerrain::TerrainPager* terrainPager );

    public:
      RtLambertianOpaqueMaterial( RtMaterialInstance instance );

      /**
      \return the surface reflectivity of this material
      */
      float getReflectivity() const;

      /**
      Assign new surface reflectivity of this material. The reflectivity
      affects the intensity output of lidars.
      \param reflectivity - new reflectivity of this material
      */
      RtLambertianOpaqueMaterial setReflectivity( float reflectivity );

    public:
      void store( agxStream::OutputArchive& out ) const;
      void restore( agxStream::InputArchive& in );

    private:
      using RtSurfaceMaterial::RtSurfaceMaterial;
  };
}
