/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/MonoaxialSignalSystemNode.h>

#include <random>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( IMonoaxialSignalNoiseNode );
  AGX_DECLARE_POINTER_TYPES( MonoaxialSignalNoiseNode );
  AGX_DECLARE_VECTOR_TYPES( MonoaxialSignalNoiseNode );

  /**
  Interface for implementation details for monoaxial signal system nodes carrying a random noise
  generator.
  */
  class AGXSENSOR_EXPORT IMonoaxialSignalNoiseNode : public IMonoaxialSignalSystemNode
  {
    public:
      /**
      Construct a trixaial signal noise node implementation.
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      IMonoaxialSignalNoiseNode( agx::UInt32 seed = 687687u );

      /**
      Set the noise seed.
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      void setSeed( agx::UInt32 seed );

      /**
      \return seed used to configure the pseudo-random perturbation generation
      */
      agx::UInt32 getSeed() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual MonoaxialSignalSystemNodeRef createMonoaxialNodeProxy() override;
      virtual void synchronize( SystemNodeProxy& proxy, const CallbackData& data ) override;
      virtual void result( SystemNodeProxy& proxy, const CallbackData& data ) override;

      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::IMonoaxialSignalNoiseNode );
      virtual void store( agxStream::OutputArchive& out ) const override;
      virtual void restore( agxStream::InputArchive& in ) override;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::UInt32 m_seed;
      bool m_seedChanged;
  };

  /**
  A system node in the sensor processing tree carrying a monoaxial signal, a random noise generator
  and an implementation for how to apply the noise.
  */
  class AGXSENSOR_EXPORT MonoaxialSignalNoiseNode : public MonoaxialSignalSystemNode
  {
    public:
      typedef std::mt19937_64 RandomGenerator;
      using result_type = RandomGenerator::result_type;

    public:
      /**
      \return minimum noise value that can be generated
      */
      inline static constexpr RandomGenerator::result_type min()
      {
        return RandomGenerator::min();
      }

      /**
      \return maximum noise value that can be generated
      */
      inline static constexpr RandomGenerator::result_type max()
      {
        return RandomGenerator::max();
      }

    public:
      /**
      Construct a monoaxial signal noise node instance with the given \p implementation.
      \param implementation - behavioral implementation for this proxy (should not be nullptr)
      */
      MonoaxialSignalNoiseNode( IMonoaxialSignalNoiseNode* implementation );

      /**
      Set the noise seed.
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      void setSeed( agx::UInt32 seed );

      /**
      \return seed used to configure the pseudo-random perturbation generation
      */
      agx::UInt32 getSeed() const;

      /**
      \return generated noise value
      */
      RandomGenerator::result_type operator()();

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      MonoaxialSignalNoiseNode();

      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::MonoaxialSignalNoiseNode );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::UInt32 m_seed;
      agx::UInt64 m_generatedCount;
      RandomGenerator m_generator;
  };
}
