/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/FrameAttachedSensor.h>
#include <agxSensor/IMUModel.h>
#include <agxSensor/IMUOutputHandler.h>

#include <agxSDK/Simulation.h>

#include <agx/Frame.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( IMU );
  AGX_DECLARE_VECTOR_TYPES( IMU );

  /**
  Inertial measurement unit (IMU) instance class defined by a frame/transform and a model. Any
  output from the sensor is accessed through the IMUOutputHandler.
  */
  class AGXSENSOR_EXPORT IMU : public FrameAttachedSensor
  {
    public:
      using SensorAttachment = IMUModelSensorAttachment::SensorAttachment;
      using SensorAttachmentVector = IMUModelSensorAttachment::SensorAttachmentVector;

    public:
      /**
      Find IMU given name.
      \param simulation - simulation with a sensor environment
      \param name - name of IMU instance to search for
      \return first IMU instance with name \p name if found, otherwise nullptr
      */
      static IMU* find( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find IMU given name.
      \param environment - sensor environment
      \param name - name of IMU instance to search for
      \return first IMU instance with name \p name if found, otherwise nullptr
      */
      static IMU* find( const Environment* environment, const agx::Name& name );

      /**
      Find all IMUs, given name.
      \param simulation - simulation with a sensor environment
      \param name - name of IMU instance to search for
      \return IMU instances with name \p name
      */
      static IMUPtrVector findAll( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find all IMUs, given name.
      \param environment - sensor environment
      \param name - name of IMU instance to search for
      \return IMU instances with name \p name
      */
      static IMUPtrVector findAll( const Environment* environment, const agx::Name& name );

      /**
      Find all IMUs.
      \param simulation - simulation with a sensor environment
      \return all IMU instances in the sensor environment
      */
      static IMUPtrVector findAll( const agxSDK::Simulation* simulation );

      /**
      Find all IMUs.
      \param environment - sensor environment
      \return all IMU instances in the given sensor environment \p environment
      */
      static IMUPtrVector findAll( const Environment* environment );

    public:
      /**
      Construct an inertial measurement unit (IMU) of the given \p model and attached to the
      specified \p frame.
      \note This IMU instance is invalid if the given \p model is nullptr.
      \param frame - parent frame to attach this IMU to, may be dynamic, the output is relative to
                     this frame. If nullptr, a new frame is created aligned to the world frame
      \param model - model defining the parameters of this IMU instance, invalid if
                     nullptr
      */
      IMU( agx::Frame* frame, IMUModel* model );

      /**
      Assign a new parent frame to attach this IMU to. The effects will take place the next time the
      environment is stepped.
      \param frame - new frame for this IMU
      */
      virtual void setFrame( agx::Frame* frame ) override;

      /**
      \return the frame of this IMU, defining this IMU -> world transform
      */
      virtual agx::Frame* getFrame() const override;

      /**
      \return the model of this IMU
      */
      IMUModel* getModel() const;

      /**
      \return the output handler managing the outputs from the IMU
      */
      IMUOutputHandler* getOutputHandler() const;

      /**
      \return the output handler managing the outputs from the IMU
      */
      virtual ISensorOutputHandler* getOutputHandlerBase() const override;

      /**
      \return list of the sensors attached onto the IMU
      */
      const SensorAttachmentVector& getSensorAttachments() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::IMU );

    protected:
      IMU();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      bool add( SensorAttachment&& sensorAttachment );

    private:
      agx::FrameRef m_frame;
      IMUModelRef m_model;
      SensorAttachmentVector m_sensorAttachments;
  };
}
