/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/EncoderModel.h>
#include <agxSensor/EncoderOutputHandler.h>
#include <agxSensor/Environment.h>
#include <agxSensor/ConstraintAttachedSensor.h>

#include <agxSDK/Simulation.h>

#include <agx/CylindricalJoint.h>
#include <agx/Hinge.h>
#include <agx/Prismatic.h>
#include <agx/SplineJoint.h>
#include <agx/SlackCylindricalJoint.h>
#include <agx/SlackHingeJoint.h>
#include <agx/SlackPrismaticJoint.h>

#include <agxVehicle/WheelJoint.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( Encoder );
  AGX_DECLARE_VECTOR_TYPES( Encoder );

  /**
  Encoder instance class defined by an attachment constraint and a model. Any output is accessed
  through the EncoderOutputHandler.
  */
  class AGXSENSOR_EXPORT Encoder : public ConstraintAttachedSensor
  {
    public:
      /**
      Find encoder given name.
      \param simulation - simulation with a sensor environment
      \param name - name of encoder instance to search for
      \return first encoder instance with name \p name if found, otherwise nullptr
      */
      static Encoder* find( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find encoder given name.
      \param environment - sensor environment
      \param name - name of encoder instance to search for
      \return first encoder instance with name \p name if found, otherwise nullptr
      */
      static Encoder* find( const Environment* environment, const agx::Name& name );

      /**
      Find all encoders, given name.
      \param simulation - simulation with a sensor environment
      \param name - name of encoder instance to search for
      \return encoder instances with name \p name
      */
      static EncoderPtrVector findAll( const agxSDK::Simulation* simulation,
                                       const agx::Name& name );

      /**
      Find all encoders, given name.
      \param environment - sensor environment
      \param name - name of encoder instance to search for
      \return encoder instances with name \p name
      */
      static EncoderPtrVector findAll( const Environment* environment, const agx::Name& name );

      /**
      Find all encoders.
      \param simulation - simulation with a sensor environment
      \return all encoder instances in the sensor environment
      */
      static EncoderPtrVector findAll( const agxSDK::Simulation* simulation );

      /**
      Find all encoders.
      \param environment - sensor environment
      \return all encoder instances in the given sensor environment \p environment
      */
      static EncoderPtrVector findAll( const Environment* environment );

    public:
      /**
      Construct this encoder with the given \p model and attached to the given cylindrical joint
      constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param cylindrical - cylindrical constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      \param sampleDof - DoF index from which the encoder should sample
      */
      Encoder( agx::CylindricalJoint* cylindrical, EncoderModel* model,
               agx::Constraint2DOF::DOF sampleDof = agx::Constraint2DOF::FIRST );

      /**
      Construct this encoder with the given \p model and attached to the given hinge constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param hinge - hinge constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      */
      Encoder( agx::Hinge* hinge, EncoderModel* model );

      /**
      Construct this encoder with the given \p model and attached to the given prismatic constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param prismatic - prismatic constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      */
      Encoder( agx::Prismatic* prismatic, EncoderModel* model );

      /**
      Construct this encoder with the given \p model and attached to the given spline constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param spline - spline constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      \param sampleDof - DoF index from which the encoder should sample
      */
      Encoder( agx::SplineJoint* spline, EncoderModel* model,
               agx::Constraint2DOF::DOF sampleDof = agx::Constraint2DOF::FIRST );

      /**
      Construct this encoder with the given \p model and attached to the given slack cylindrical
      joint constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param cylindrical - slack cylindrical constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      \param sampleDof - DoF index from which the encoder should sample
      */
      Encoder( agx::SlackCylindricalJoint* cylindrical, EncoderModel* model,
               agx::Constraint2DOF::DOF sampleDof = agx::Constraint2DOF::FIRST );

      /**
      Construct this encoder with the given \p model and attached to the given slack hinge
      constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param hinge - slack hinge constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      */
      Encoder( agx::SlackHingeJoint* hinge, EncoderModel* model );

      /**
      Construct this encoder with the given \p model and attached to the given slack prismatic
      constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param prismatic - slack prismatic constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      */
      Encoder( agx::SlackPrismatic* prismatic, EncoderModel* model );

      /**
      Construct this encoder with the given \p model and attached to the given wheel joint
      constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param wheel - wheel constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      \param sampleDof - DoF index from which the encoder should sample
      */
      Encoder( agxVehicle::WheelJoint* wheel, EncoderModel* model,
               agxVehicle::WheelJoint::SecondaryConstraint sampleDof
                 = agxVehicle::WheelJoint::WHEEL );

      /**
      Construct this encoder with the given \p model and attached to the given constraint.
      \note This encoder instance is invalid if the given \p model is nullptr.
      \param constraint - constraint to attach to and sample from
      \param model - model defining the parameters of this encoder instance, invalid if nullptr
      \param sampleDof - DoF index from which the encoder should sample
      */
      Encoder( agx::Constraint* constraint, EncoderModel* model, agx::UInt sampleDof );

      /**
      Assign a new parent constraint to attach this encoder to. The effects will take place the next
      time the environment is stepped.
      \param constraint - new constraint for this encoder
      */
      virtual void setConstraint( agx::Constraint* constraint ) override;

      /**
      \return the constraint of this encoder, defining what the encoder is attached to
      */
      virtual agx::Constraint* getConstraint() const override;

      /**
      Set the DoF in the constraint form which the encoder samples the positional value.
      \param dof - DoF index in the constraint to sample
      */
      void setConstraintSampleDof( agx::UInt dof );

      /**
      \return DoF index in the constraint from  which the encoder samples the positional value
      */
      agx::UInt getConstraintSampleDof() const;

      /**
      \return the model of this encoder
      */
      EncoderModel* getModel() const;

      /**
      \return the output handler managing the outputs from the encoder
      */
      EncoderOutputHandler* getOutputHandler() const;

      /**
      \return the output handler managing the outputs from the encoder
      */
      virtual ISensorOutputHandler* getOutputHandlerBase() const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::Encoder );

    protected:
      Encoder();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::ConstraintRef m_constraint;
      agx::UInt m_constraintSampleDof;
      EncoderModelRef m_model;
  };
}
