/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <openplx/ErrorReporter.h>
#include <openplx/Physics/Geometries/ContactGeometry.h>
#include <openplx/Physics/System.h>
#include <openplx/Sensors/AccelerometerLogic.h>
#include <openplx/Sensors/GyroscopeLogic.h>
#include <openplx/Sensors/MagnetometerLogic.h>
#include <openplx/Sensors/TriaxialRange.h>
#include <openplx/Sensors/Optics/RaySource.h>
#include <openplx/Sensors/PulsedLidarLogic.h>
#include <openplx/Sensors/Signals/AccelerometerOutput.h>
#include <openplx/Sensors/Signals/GyroscopeOutput.h>
#include <openplx/Sensors/Signals/MagnetometerOutput.h>
#include <openplx/Sensors/Signals/LidarOutput.h>
#include <openplx/Visuals/Geometries/Geometry.h>

#include <agxSensor/Accelerometer.h>
#include <agxSensor/Environment.h>
#include <agxSensor/Gyroscope.h>
#include <agxSensor/Lidar.h>
#include <agxSensor/Magnetometer.h>
#include <agxSensor/RaytraceDistanceGaussianNoise.h>
#include <agxSensor/RaytraceGgxAndOrenNayarMaterial.h>
#include <agxSensor/RaytraceLambertianOpaqueMaterial.h>
#include <agxOpenPLX/AgxMetadata.h>
#include <agxOpenPLX/export.h>

namespace agxopenplx
{
  class OpenPlxToAgxMapper;
  class OpenPlxToAgxVisualsMapper;

  /**
   * @brief Metadata stored for a lidar sensor.
   */
  struct OpenPlxSensorsLidarMetadata : public AgxMetadataDynamic
  {
      agx::Real wavelength;
      std::unordered_map<const openplx::Sensors::Signals::LidarOutput*, agxSensor::RtOutputRef> output_mapping;
  };

  /**
   * @brief Metadata stored for a triaxial sensor.
   */
  struct OpenPlxSensorsAccelerometerMetadata : public AgxMetadataDynamic
  {
    std::unordered_map<const openplx::Sensors::Signals::AccelerometerOutput*, agxSensor::TriaxialOutputRef> output_mapping;
  };

  /**
   * @brief Metadata stored for a gyroscope sensor.
   */
  struct OpenPlxSensorsGyroscopeMetadata : public AgxMetadataDynamic
  {
    std::unordered_map<const openplx::Sensors::Signals::GyroscopeOutput*, agxSensor::TriaxialOutputRef> output_mapping;
  };

  /**
   * @brief Metadata stored for a magnetometer sensor.
   */
  struct OpenPlxSensorsMagnetometerMetadata : public AgxMetadataDynamic
  {
    std::unordered_map<const openplx::Sensors::Signals::MagnetometerOutput*, agxSensor::TriaxialOutputRef> output_mapping;
  };

  /**
   * @brief Maps the content of the OpenPLX sensor bundle to its analogues in AGX Dynamics.
   */
  class AGXOPENPLX_EXPORT OpenPlxSensorsMapper
  {
    public:
      // ID of the first lidar output, the rest are mapped with decreasing IDs from this value.
      constexpr static std::size_t FIRST_LIDAR_OUTPUT_ID = 0xFFFFFFFFFFFFFFFF;
      // ID of the first accelerometer output, the rest are mapped with decreasing IDs from this value.
      constexpr static std::size_t FIRST_ACCELEROMETER_OUTPUT_ID = 0xFFFFFFFFFFFFFFFF;
      // ID of the first gyroscope output, the rest are mapped with decreasing IDs from this value.
      constexpr static std::size_t FIRST_GYROSCOPE_OUTPUT_ID = 0xFFFFFFFFFFFFFFFF;
      // ID of the first magnetometer output, the rest are mapped with decreasing IDs from this value.
      constexpr static std::size_t FIRST_MAGNETOMETER_OUTPUT_ID = 0xFFFFFFFFFFFFFFFF;

    public:
      OpenPlxSensorsMapper(
        std::shared_ptr<openplx::ErrorReporter> error_reporter, std::shared_ptr<AgxMetadata> metadata);

    public:
      const std::unordered_map<std::shared_ptr<openplx::Sensors::LidarLogic>, agxSensor::LidarRef>&
      getMappedLidarLogics() const;

    public:
      void mapSensorsIntoEnvironment(
        const std::shared_ptr<openplx::Physics::System>& system, agxSensor::Environment* environment,
        OpenPlxToAgxMapper& openplx_to_agx_mapper);
      void mapGeometriesIntoEnvironment(
        const std::shared_ptr<openplx::Physics::System>& system, agxSensor::Environment* environment,
        OpenPlxToAgxMapper& openplx_to_agx_mapper);
      void mapGeometriesIntoEnvironment(
        const std::shared_ptr<openplx::Physics::System>& system, agxSensor::Environment* environment,
        OpenPlxToAgxMapper& openplx_to_agx_mapper, OpenPlxToAgxVisualsMapper& openplx_to_agx_visuals_mapper);

    private:
      // Optics
      agxSensor::LidarPropertiesRef mapConicalBeamDivergenceToProperties(
        const std::shared_ptr<openplx::Sensors::Optics::ConicalBeamDivergence>& conical_beam_divergence);
      agxSensor::LidarPropertiesRef mapBeamDivergenceToProperties(
        const std::shared_ptr<openplx::Sensors::Optics::BeamDivergence>& beam_divergence);

      agxSensor::LidarRayPatternGeneratorRef mapHorizontalSweepRaySourceToPatternGenerator(
        const std::shared_ptr<openplx::Sensors::Optics::HorizontalSweepRaySource>& horizontal_sweep_ray_source);
      agxSensor::LidarRayPatternGeneratorRef mapRaySourceToPatternGenerator(
        const std::shared_ptr<openplx::Sensors::Optics::RaySource>& ray_source);

      agxSensor::LidarRayDistortionRef mapRayEmissionAngleGaussianNoiseToRayDistortion(
        const std::shared_ptr<openplx::Sensors::Optics::RayEmissionAngleGaussianNoise>& ray_angle_gaussian_noise);
      agxSensor::LidarRayDistortionRef mapRayEmissionDistortionToRayDistortion(
        const std::shared_ptr<openplx::Sensors::Optics::RayEmissionDistortion>& ray_emission_distortion);

      // Lidar
      void mapLidarLogicRayEmissionDistortions(
        const std::vector<std::shared_ptr<openplx::Sensors::Optics::RayEmissionDistortion>>& ray_emission_distortions,
        agxSensor::LidarRayDistortionHandler& ray_distortion_handler);

      agxSensor::RtOutputNoiseRef mapLidarDetectionDistanceGaussianNoise(
        const std::shared_ptr<openplx::Sensors::LidarDetectionDistanceGaussianNoise>& distance_gaussian_noise);
      agxSensor::RtOutputNoiseRef mapLidarSensingDistortion(
        const std::shared_ptr<openplx::Sensors::LidarSensingDistortion>& lidar_sensing_distortion);
      agxSensor::RtOutputNoiseRefVector mapLidarLogicSensingDistortions(
        const std::vector<std::shared_ptr<openplx::Sensors::LidarSensingDistortion>>& sensing_distortions);

      std::optional<agxSensor::RtOutput::Field> getLidarOutputFieldFor(std::int64_t openplx_field);
      agxSensor::RtOutputRef mapLidarOutput(const openplx::Sensors::Signals::LidarOutput& lidar_output);
      void mapLidarLogicOutputs(
        const std::shared_ptr<openplx::Sensors::LidarLogic>& lidar_logic, agxSensor::LidarRef agx_lidar,
        std::unordered_set<std::size_t>& used_output_ids);

      agxSensor::LidarRef mapPulsedLidar(
        const std::shared_ptr<openplx::Sensors::PulsedLidarLogic>& pulsed_lidar_logic,
        const std::string& pulsed_lidar_logic_key, OpenPlxToAgxMapper& openplx_to_agx_mapper);

      // Accelerometer
      agxSensor::TriaxialRange mapTriaxialRange(
        const std::shared_ptr<openplx::Sensors::TriaxialRange>& triaxial_range);

      agxSensor::ITriaxialSignalSystemNodeRef mapAccelerometerSignalGaussianNoise(
        const std::shared_ptr<openplx::Sensors::AccelerometerSignalGaussianNoise>& gaussian_noise);
      agxSensor::ITriaxialSignalSystemNodeRef mapAccelerometerSignalSpectralGaussianNoise(
        const std::shared_ptr<openplx::Sensors::AccelerometerSignalSpectralGaussianNoise>& spectral_gaussian_noise);
      agxSensor::ITriaxialSignalSystemNodeRef mapAccelerometerSensingDistortion(
        const std::shared_ptr<openplx::Sensors::AccelerometerSensingDistortion>& accelerometer_sensing_distortion);
      agxSensor::ITriaxialSignalSystemNodeRefVector mapAccelerometerLogicSensingDistortions(
        const std::vector<std::shared_ptr<openplx::Sensors::AccelerometerSensingDistortion>>& sensing_distortions);

      std::optional<agxSensor::TriaxialOutput::Field> getAccelerometerOutputFieldFor(std::int64_t openplx_field);
      agxSensor::TriaxialOutputRef mapAccelerometerOutput(const openplx::Sensors::Signals::AccelerometerOutput& accelerometer_output);
      void mapAccelerometerLogicOutputs(
        const std::shared_ptr<openplx::Sensors::AccelerometerLogic>& accelerometer_logic, agxSensor::AccelerometerRef agx_accelerometer,
        std::unordered_set<std::size_t>& used_output_ids);

      agxSensor::AccelerometerRef mapAccelerometer(
        const std::shared_ptr<openplx::Sensors::AccelerometerLogic>& accelerometer_logic, const std::string& accelerometer_logic_key,
        OpenPlxToAgxMapper& openplx_to_agx_mapper);

      //  Gyroscope
      agxSensor::ITriaxialSignalSystemNodeRef mapGyroscopeSignalGaussianNoise(
        const std::shared_ptr<openplx::Sensors::GyroscopeSignalGaussianNoise>& gaussian_noise);
      agxSensor::ITriaxialSignalSystemNodeRef mapGyroscopeSignalSpectralGaussianNoise(
        const std::shared_ptr<openplx::Sensors::GyroscopeSignalSpectralGaussianNoise>& spectral_gaussian_noise);
      agxSensor::ITriaxialSignalSystemNodeRef mapGyroscopeSignalLinearAccelectionEffect(
        const std::shared_ptr<openplx::Sensors::GyroscopeSignalLinearAccelectionEffect>& linear_acceleration_effect);
      agxSensor::ITriaxialSignalSystemNodeRef mapGyroscopeSensingDistortion(
        const std::shared_ptr<openplx::Sensors::GyroscopeSensingDistortion>& gyroscope_sensing_distortion);
      agxSensor::ITriaxialSignalSystemNodeRefVector mapGyroscopeLogicSensingDistortions(
        const std::vector<std::shared_ptr<openplx::Sensors::GyroscopeSensingDistortion>>& sensing_distortions);

      std::optional<agxSensor::TriaxialOutput::Field> getGyroscopeOutputFieldFor(std::int64_t openplx_field);
      agxSensor::TriaxialOutputRef mapGyroscopeOutput(const openplx::Sensors::Signals::GyroscopeOutput& gyroscope_output);
      void mapGyroscopeLogicOutputs(
        const std::shared_ptr<openplx::Sensors::GyroscopeLogic>& gyroscope_logic, agxSensor::GyroscopeRef agx_gyroscope,
        std::unordered_set<std::size_t>& used_output_ids);

      agxSensor::GyroscopeRef mapGyroscope(
        const std::shared_ptr<openplx::Sensors::GyroscopeLogic>& gyroscope_logic, const std::string& gyroscope_logic_key,
        OpenPlxToAgxMapper& openplx_to_agx_mapper);

      // Magnetometer
      agxSensor::ITriaxialSignalSystemNodeRef mapMagnetometerSignalGaussianNoise(
        const std::shared_ptr<openplx::Sensors::MagnetometerSignalGaussianNoise>& gaussian_noise);
      agxSensor::ITriaxialSignalSystemNodeRef mapMagnetometerSignalSpectralGaussianNoise(
        const std::shared_ptr<openplx::Sensors::MagnetometerSignalSpectralGaussianNoise>& spectral_gaussian_noise);
      agxSensor::ITriaxialSignalSystemNodeRef mapMagnetometerSensingDistortion(
        const std::shared_ptr<openplx::Sensors::MagnetometerSensingDistortion>& magnetometer_sensing_distortion);
      agxSensor::ITriaxialSignalSystemNodeRefVector mapMagnetometerLogicSensingDistortions(
        const std::vector<std::shared_ptr<openplx::Sensors::MagnetometerSensingDistortion>>& sensing_distortions);

      std::optional<agxSensor::TriaxialOutput::Field> getMagnetometerOutputFieldFor(std::int64_t openplx_field);
      agxSensor::TriaxialOutputRef mapMagnetometerOutput(const openplx::Sensors::Signals::MagnetometerOutput& magnetometer_output);
      void mapMagnetometerLogicOutputs(
        const std::shared_ptr<openplx::Sensors::MagnetometerLogic>& magnetometer_logic, agxSensor::MagnetometerRef agx_magnetometer,
        std::unordered_set<std::size_t>& used_output_ids);

      agxSensor::MagnetometerRef mapMagnetometer(
        const std::shared_ptr<openplx::Sensors::MagnetometerLogic>& magnetometer_logic, const std::string& magnetometer_logic_key,
        OpenPlxToAgxMapper& openplx_to_agx_mapper);

      // Geometries
      agx::Vec2f mapComplexRefractiveIndex(const std::shared_ptr<openplx::Core::Object>& with_refractive_index);

      agxSensor::RtGgxAndOrenNayarMaterial mapGgxAndOrenNayarSurface(
        const std::shared_ptr<openplx::Core::Object>& with_ggx_oren_nayar_model);
      agxSensor::RtGgxAndOrenNayarMaterial mapGgxSurface(const std::shared_ptr<openplx::Core::Object>& with_ggx_model);
      agxSensor::RtGgxAndOrenNayarMaterial mapOrenNayarSurface(
        const std::shared_ptr<openplx::Core::Object>& with_oren_nayar_model);
      agxSensor::RtLambertianOpaqueMaterial mapLambertSurface(
        const std::shared_ptr<openplx::Core::Object>& with_lambert_model);

      void mapGeometryOpticsMaterial(
        agxCollide::GeometryRef geometry, const std::shared_ptr<openplx::Physics::Optics::Material>& material);

      void mapGeometryIntoEnvironment(
        agxSensor::Environment* environment, const std::shared_ptr<openplx::Visuals::Geometries::Geometry>& geometry,
        const std::string& geometry_key, const OpenPlxToAgxVisualsMapper& openplx_to_agx_visuals_mapper);
      void mapGeometryIntoEnvironment(
        agxSensor::Environment* environment,
        const std::shared_ptr<openplx::Physics::Geometries::ContactGeometry>& geometry, const std::string& geometry_key,
        const OpenPlxToAgxMapper& openplx_to_agx_mapper);

      // Misc
      agx::FrameRef mapSensorMateConnector(openplx::Core::ObjectPtr mate_connector_owner,
        const std::shared_ptr<openplx::Physics3D::Interactions::MateConnector>& mate_connector, OpenPlxToAgxMapper& openplx_to_agx_mapper);

    private:
      std::shared_ptr<openplx::ErrorReporter> m_error_reporter;
      std::shared_ptr<AgxMetadata> m_metadata;
      std::shared_ptr<std::unordered_map<std::int64_t, agxSensor::RtOutput::Field>> m_lidar_field_map;
      std::shared_ptr<std::unordered_map<std::int64_t, agxSensor::TriaxialOutput::Field>> m_accelerometer_field_map;
      std::shared_ptr<std::unordered_map<std::int64_t, agxSensor::TriaxialOutput::Field>> m_gyroscope_field_map;
      std::shared_ptr<std::unordered_map<std::int64_t, agxSensor::TriaxialOutput::Field>> m_magnetometer_field_map;

      std::unordered_map<std::shared_ptr<openplx::Sensors::LidarLogic>, agxSensor::LidarRef> m_mapped_lidar_logics;
      std::unordered_map<std::shared_ptr<openplx::Sensors::AccelerometerLogic>, agxSensor::AccelerometerRef> m_mapped_accelerometer_logics;
      std::unordered_map<std::shared_ptr<openplx::Sensors::GyroscopeLogic>, agxSensor::GyroscopeRef> m_mapped_gyroscope_logics;
      std::unordered_map<std::shared_ptr<openplx::Sensors::MagnetometerLogic>, agxSensor::MagnetometerRef> m_mapped_magnetometer_logics;

  };
}
