#pragma once

#include <unordered_set>
#include <openplx/NodeVisitor.h>
#include <openplx/DocumentContext.h>
#include <openplx/ErrorReporter.h>
#include <openplx/TreeNode.h>

namespace openplx {
    class SymbolTreeBuilder {
        public:

            static bool build(const std::vector<std::shared_ptr<Document>>& documents,
                              const std::shared_ptr<ErrorReporter>& error_reporter);

            static bool build(const std::vector<std::shared_ptr<DocumentContext>>& documents,
                              const std::shared_ptr<ErrorReporter>& error_reporter);

            static bool build(const std::unordered_set<std::shared_ptr<Node>>& nodes,
                              const std::shared_ptr<ErrorReporter>& error_reporter);

            static void buildInnerTree(const ModelDeclPtr& model_declaration);
            static void validateInnerTree(const ModelDeclPtr& model_declaration,
                                          const std::shared_ptr<ErrorReporter>& error_reporter);

            /**
             * extendTypesAndValues adds and overwrites symbol meta data in outer_root by
             * adding everything in the source and the outer assignments in the inner roots.
             *
             * Example 1:
             *
             * Foo:
             *     x is Real: 1.0
             *
             * Passing an empty outer_root for Foo, source as nullptr and the Foo inner tree
             * as the single inner root in inner_roots produces a complete symbol tree for Foo.
             *
             * Example 2:
             *
             * Foo2 is Foo:
             *     y is Real: 2.0
             *
             * Passing an empty outer_root, source as the outer tree for Foo and
             * the inner tree for Foo2 produces a complete symbol tree for Foo2.
             *
             * Example 3:
             *
             * trait T1:
             *     x is Real: 1.0
             *
             * Foo:
             *     with T1
             *     y is Real: 20
             *
             * Passing an empty outer_root, nullptr as source, T1 as a single trait in *traits*
             * and the inner root of Foo in inner_roots. Produces a complete symbol tree for Foo.
             */
            static void extendTypesAndValues(
                const std::shared_ptr<ErrorReporter>& error_reporter,
                const std::shared_ptr<TreeNode<OuterSymbol>>& outer_root,
                const std::shared_ptr<TreeNode<OuterSymbol>>& source,
                const std::vector<std::shared_ptr<ModelDeclaration>>& traits,
                const std::vector<std::shared_ptr<TreeNode<InnerSymbol>>>& inner_roots,
                size_t depth = 0);

            static void extendPaths(const std::shared_ptr<TreeNode<OuterSymbol>>& outer_root);
            static void setRootModel(const ModelDeclPtr& root_model);
    };
}
