#pragma once

#include <memory>
#include <string>
#include <optional>
#include <unordered_map>

namespace openplx {

    namespace Core {
        class Object;
    }

    enum class FieldType {
        Bool = 0,
        UInt = 1,
        Int  = 2,
        Real = 3,
        Max
    };

    enum class FieldArrayType {
        None     = 0, // Int
        Static   = 1, // Int[3], NOT SUPPORTED
        Dynamic  = 2, // Int[size], NOT SUPPORTED
        Flexible = 3, // Int[]
        Max
    };

    struct Field {
        FieldType field_type;
        size_t size;
        size_t offset;
        FieldArrayType field_array_type;
    };

    class Marshalling {
        public:
            Marshalling();
            static std::optional<Field> field_from_object(const std::shared_ptr<Core::Object>& object);
            bool add_fields_from_object(const std::shared_ptr<Core::Object>& object);
            void add_field(std::string name, FieldType field_type, size_t size, size_t offset, FieldArrayType field_array_type = FieldArrayType::None);
            std::unique_ptr<Marshalling>& get_or_add_nested_marshalling(const std::string& name);

            void set_buffer(uint8_t* buffer, size_t buffer_size);
            void set_array_type(FieldArrayType array_type);
            FieldArrayType get_array_type();

            template <typename T>
            std::optional<T> read_real(const std::string& field_name);

            template <typename T>
            std::optional<T> read_real_indexed(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            T read_real_unsafe(const std::string& field_name);

            template <typename T>
            T read_real_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            std::optional<T> read_int(const std::string& field_name);

            template <typename T>
            std::optional<T> read_int_indexed(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            T read_int_unsafe(const std::string& field_name);

            template <typename T>
            T read_int_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            std::optional<T> read_uint(const std::string& field_name);

            template <typename T>
            std::optional<T> read_uint_indexed(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            T read_uint_unsafe(const std::string& field_name);

            template <typename T>
            T read_uint_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            std::optional<bool> read_bool(const std::string& field_name);
            std::optional<bool> read_bool_indexed(const std::string& field_name, size_t index, size_t field_index);
            bool read_bool_unsafe(const std::string& field_name);
            bool read_bool_indexed_unsafe(const std::string& field_name, size_t index, size_t field_index);

            template <typename T>
            bool write_real(const std::string& field_name, T value);

            template <typename T>
            bool write_real_indexed(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            void write_real_unsafe(const std::string& field_name, T value);

            template <typename T>
            void write_real_indexed_unsafe(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            bool write_int(const std::string& field_name, T value);

            template <typename T>
            bool write_int_indexed(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            void write_int_unsafe(const std::string& field_name, T value);

            template <typename T>
            void write_int_indexed_unsafe(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            bool write_uint(const std::string& field_name, T value);

            template <typename T>
            bool write_uint_indexed(const std::string& field_name, T value, size_t index, size_t field_index);

            template <typename T>
            void write_uint_unsafe(const std::string& field_name, T value);

            template <typename T>
            void write_uint_indexed_unsafe(const std::string& field_name, T value, size_t index, size_t field_index);

            bool write_bool(const std::string& field_name, bool value);
            bool write_bool_indexed(const std::string& field_name, bool value, size_t index, size_t field_index);
            void write_bool_unsafe(const std::string& field_name, bool value);
            void write_bool_indexed_unsafe(const std::string& field_name, bool value, size_t index, size_t field_index);


        private:
            bool check_indices(Field& field, size_t index, size_t field_index);

            size_t m_size;
            size_t m_offset;
            FieldArrayType m_array_type;
            uint8_t* m_buffer;
            size_t m_buffer_size;
            std::unordered_map<std::string, Field> m_field_map;
            std::unordered_map<std::string, std::unique_ptr<Marshalling>> m_nested_map;
    };
}
