#pragma once
#include <memory>
#include <variant>

namespace openplx
{
    /**
     * @brief A blob of binary data, either owning the data or referencing a block of memory owned by someone else.
     */
    class Blob {
        public:
            /**
             * @brief An empty blob.
             */
            Blob();

            /**
             * @brief Create a blob referencing memory.
             * @param size Size of the referenced block of memory.
             * @param data Reference pointer to a block of memory.
             */
            explicit Blob(size_t size, void* data);

            /**
             * @brief Create a blob owning memory.
             * @param size Size of the owned block of memory.
             * @param data Owned block of memory.
             */
            explicit Blob(size_t size, std::unique_ptr<char[]>&& data);

            // YES!Move
            Blob(Blob&&) noexcept = default;
            Blob& operator=(Blob&&) noexcept = default;

            // NO!Copy
            Blob(const Blob&) = delete;
            Blob& operator=(const Blob&) = delete;

        public:
            /**
             * @return True if this blob is a reference to memory owned by someone else.
             */
            bool isReference() const;

            /**
             * @return Size of the blob memory.
             */
            size_t getSize() const;

            /**
             * @return Pointer to the blob memory.
             */
            void* getData() const;

            /**
             * @brief Attempts to detach the data from this blob and return it.
             * @return The owned data or nullptr if this is a reference blob.
             */
            std::unique_ptr<char[]> releaseData();

        private:
            size_t m_size;
            std::variant<std::unique_ptr<char[]>, void*> m_data;
    };

}
