/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/MagneticField.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( UniformMagneticField );

  /**
  A magnetic field with equal direction and strength across the entire sensor environment.
  Default uniform magnetic field vector is set as the relative magnetic field vector on Earth at
  the surface near the coordinates of 38.254113 degrees north and 140.875417 degrees east, based on
  World Magnetic Model values in early 2025.
  */
  class AGXSENSOR_EXPORT UniformMagneticField : public MagneticField
  {
    public:
      /**
      Constructs a uniform magnetic field with the specified \p magneticField vector.
      \param magneticField - magnetic field vector
      */
      UniformMagneticField( const agx::Vec3& magneticField = agx::Vec3( -19.462e-6,
                                                                        44.754e-6,
                                                                        7.8426e-6 ) );

      /**
      Sets the uniform magnetic field vector.
      \param magneticField - magnetic field vector
      */
      void setMagneticField( const agx::Vec3& magneticField );

      /**
      \return the magnetic field vector defining this uniform magnetic field
      */
      const agx::Vec3& getMagneticField() const;

      /**
      Calculates and returns the magnetic field at the given \p position in world coordinates.
      \param position - position in world coordinates at which the magnetic field should be
                        determined
      \return magnetic field vector at the \p position
      */
      virtual agx::Vec3 calculateMagneticField( const agx::Vec3& position ) const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::UniformMagneticField );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Vec3 m_magneticField;
  };
}
