/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/Sensor.h>

#include <array>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( SensorGroupStepStride );
  AGX_DECLARE_VECTOR_TYPES( SensorGroupStepStride );

  /**
  A group of sensors that steps at a lower rate compared to the environment, specified by the stride
  value.
  \note This group only has an effect when added to the environment, and is ignored when added as a
  child to another SystemNode.
  \note Any sensor added to this group will only produce output at the lower rate at which this
  group steps, and thus not at the full environment step rate.
  */
  class AGXSENSOR_EXPORT SensorGroupStepStride : public SystemNode
  {
    public:
      /**
      Construct a step stride sensor group with the given \p stride.
      \param stride - Number of environment steps between each step of this group
      \param sensors - Initial sensors to add to this group
      */
      SensorGroupStepStride( agx::UInt32 stride = 1,
                             SensorRefVector sensors = SensorRefVector() );

      /**
      Set the group step stride compared to the environment.
      \param stride - Number of environment steps between each step of this group
      */
      void setStride( agx::UInt32 stride );

      /**
      \return the number of environment steps between each step of this group
      */
      agx::UInt32 getStride() const;

      /**
      Add unique sensor to this group.
      \param sensor - sensor to add
      \return true if added successfully, false if nullptr or already added
      */
      bool add( Sensor* sensor );

      /**
      Remove sensor from this group.
      \param sensor - sensor to remove
      \return  true if removed successfully, false if nullptr or not sensor in this group
      */
      bool remove( Sensor* sensor );

      /**
      The sensor environment is being put down, clean everything.
      */
      virtual void cleanup() override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::SensorGroupStepStride );

    protected:
      void invoke( const CallbackEvent& callbackEvent ) override;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      struct AccumulatedStep
      {
        agx::UInt32 step;
        CallbackData data;
      };

    private:
      agx::UInt32 m_stride;
      std::array<AccumulatedStep, CallbackEvent::CLEANUP + 1> m_accumulatedSteps;
  };
}
