/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/export.h>
#include <agxSensor/TriaxialCrossSensitivity.h>
#include <agxSensor/TriaxialRange.h>
#include <agxSensor/TriaxialSignalSystemNode.h>

#include <agxStream/Serializable.h>

#include <agx/Referenced.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( AccelerometerModel );

  /**
  Base accelerometer model describing the fundamental parameters, such as measurement range and
  zero bias, of an accelerometer.
  \note The configuration in this model is shared and synchronized across all instances.
  */
  class AGXSENSOR_EXPORT AccelerometerModel : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Creates an accelerometer model outputting ideal measurement data (ground truth values).
      \return new ideal accelerometer model
      */
      static AccelerometerModel* makeIdealModel();

    public:
      /**
      Constructs an accelerometer model with the specified parameters.
      \param range - detectable range of acceleration, in m/s^2
      \param crossSensitivity - cross axis sensitivity
      \param zeroGBias - bias acceleration reported in each axis under conditions without externally
                         applied acceleration, in m/s^2
      \param outputModifiers - list of modifiers to apply to the output of the accelerometer
      */
      AccelerometerModel( TriaxialRange range = agx::RangeReal(
                            std::numeric_limits<agx::Real>::lowest(),
                            std::numeric_limits<agx::Real>::max() ),
                          TriaxialCrossSensitivity crossSensitivity = 0.01,
                          agx::Vec3 zeroGBias = agx::Vec3( 260.0 ),
                          ITriaxialSignalSystemNodeRefVector outputModifiers = ITriaxialSignalSystemNodeRefVector() );

      /**
      Set the detectable range for the accelerometer.
      \param range - detectable range of acceleration, in m/s^2
      */
      void setRange( const TriaxialRange& range );

      /**
      \return the range of acceleration, in m/s^2, that the accelerometer can detect.
      */
      const TriaxialRange& getRange() const;

      /**
      Set the accelerometer cross-axis sensitivity.
      \param crossSensitivity - cross axis sensitivity
      */
      void setCrossAxisSensitivity( const TriaxialCrossSensitivity& crossSensitivity );

      /**
      \return the cross sensitivity, describing the effects that acceleration along one axis has on
              every other axis.
      */
      const TriaxialCrossSensitivity& getCrossAxisSensitivity() const;

      /**
      Set the bias in acceleration at zero externally applied acceleration.
      \param zeroGBias - bias acceleration reported in each axis under conditions without externally
                         applied acceleration, in m/s^2
      */
      void setZeroGBias( const agx::Vec3& zeroGBias );

      /**
      \return the bias acceleration, in m/s^2, reported in each axis at under conditions without
              externally applied acceleration.
      */
      const agx::Vec3& getZeroGBias() const;

      /**
      \return the list of output modifiers to apply to the output
      */
      const ITriaxialSignalSystemNodeRefVector& getOutputModifiers() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~AccelerometerModel() = default;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::AccelerometerModel );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      TriaxialRange m_range;
      TriaxialCrossSensitivity m_crossAxisSensitivity;
      agx::Vec3 m_zeroGBias;
      ITriaxialSignalSystemNodeRefVector m_outputModifiers;
  };
}
