/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/config/AGX_USE_WEB_DEBUGGER.h>
#include <agxRender/Color.h>
#include <agxRender/RenderManager.h>
#include <agxRender/RenderProxy.h>

#if AGX_USE_WEB_DEBUGGER()

  #include <agxCollide/agxcollide_vector_types.h>



namespace agxSDK
{
  #ifndef SWIG

  /// Specifies which type of Message we are sending to the remote client.
  enum class MessageType
  {
    CREATE = 0,                /**< Create an object. */
    CREATE_MESH = 1,           /**< Create a mesh object. */
    TRANSFORM = 2,             /**< Update transform */
    REMOVE = 3,                /**< Remove an object. */
    UPDATE = 4,                /**< Update the size of an object. */
    UPDATE_MESH = 5,           /**< Update the mesh for an object. */
    COLOR = 6,                 /**< Update the color */
    CREATE_GRANULAR = 7,       /**< Create granular objects. */
    UPDATE_GRANULAR = 8,       /**< Update granular objects */
    AGX_ARCHIVE = 9,           /**< Send an AGX Archive */
    WEB_DEBUGGER_VERSION = 10, /**< Send the current version of the webdebugger protocol */
    UNDEFINED
  };

  /// Define which category of object we are creating
  enum class ObjectType
  {
    RIGIDBODY = 0, /**< A shape with a geometry which belongs to a rigid body. */
    GEOMETRY = 1,  /**< A shape with a geometry */
    GRANULAR = 2,  /**< Granular objects */
    TEXT = 3,      /**< 2D Text. */
    OTHER = 4      /**< Something else, typically debug rendering shapes (without Geometry) */
  };

  /// Defines which shape we are creating
  enum class ShapeType
  {
    SPHERE = 0,          /**< Create a sphere. */
    BOX = 1,             /**< Create a box. */
    CYLINDER = 2,        /**< Create a cylinder. */
    CAPSULE = 3,         /**< Create a capsule. */
    CONE = 4,            /**< Create a Cone. */
    LINE = 5,            /**< Create a line. */
    PLANE = 6,           /**< Create a plane. */
    MESH = 7,            /**< Create a mesh (convex or trimesh). */
    TEXT = 8,            /**< Create 2D Text. */
    HOLLOW_CYLINDER = 9, /**< Create a hollow cylinder. */
    HOLLOW_CONE = 10,    /**< Create a hollow cone. */
    TRUNCATED_CONE = 11, /**< Create a truncated cone. */
    HEIGHTFIELD = 12,    /**< Create a heightfield. */
    WIRE = 13,           /**< Create a wire shape. */
    CONTACTS = 14,       /**< Create contacts. */
    RIGIDBODIES = 15,    /**< Create rigid bodies. */
    GRANULAR = 16        /**< Create a granular object. */
  };

  /// Generic Message that can hold all data. Used when queuing a message
  struct Message
  {
    Message() noexcept;

    Message(const Message&) = default;
    Message(Message&&) noexcept = default;
    Message& operator=(const Message&) = default;
    Message& operator=(Message&&) noexcept = default;
    ~Message() = default;

    unsigned int uuid;
    float time;
    agx::UInt8 type;
    agx::UInt16 geometry;
    agx::UInt16 objectType;
    std::vector<float> color;
    std::vector<float> size;
    std::vector<double> position;
    std::vector<float> rotation;
    std::vector<float> vertices;
    std::vector<agx::UInt32> indices;
    std::string text;
    std::string name;

    std::string data;
  };



  namespace RemoteShapes
  {
    // Hold a material definition for a Remote shape.
    class Material
    {
      public:
        Material(agxRender::Color c = agxRender::Color(-1, -1, -1))
          : color(c)
        {
        }

        agxRender::Color color;
    };

    /// Baseclass for shapes that can be created in the client.
    class Shape
    {
      public:
        Shape(ShapeType _shape)
          : shapeType(_shape)
        {
        }

        /**
        Pure virtual method that will put together a message containing shape specific information.
        \param msg - The message that will be populated with data.
        */
        virtual void createMsg(Message& msg) const = 0;

        ShapeType shapeType;

        virtual ~Shape() {}
    };

    class Sphere : public Shape
    {
      public:
        Sphere(float radius_ = 0.5f)
          : Shape(ShapeType::SPHERE)
          , radius(radius_)
        {
        }

        void createMsg(Message& msg) const override;

        float radius;
    };


    class Box : public Shape
    {
      public:
        Box(agx::Vec3 size_ = agx::Vec3(0.5))
          : Shape(ShapeType::BOX)
          , size(size_)
        {
        }

        void createMsg(Message& msg) const override;

        agx::Vec3 size;
    };

    class Text : public Shape
    {
      public:
        Text(const std::string& text_ = "", float size_ = 1)
          : Shape(ShapeType::TEXT)
          , text(text_)
          , size(size_)
        {
        }

        void createMsg(Message& msg) const override;

        std::string text;
        float size;
        agx::Vec3 pos;
    };

    class Cylinder : public Shape
    {
      public:
        Cylinder(float radius_ = 0.5f, float length_ = 1)
          : Shape(ShapeType::CYLINDER)
          , radius(radius_)
          , length(length_)
        {
        }
        void createMsg(Message& msg) const override;

        float radius;
        float length;
    };

    class Capsule : public Shape
    {
      public:
        Capsule(float radius_ = 0.5f, float length_ = 1)
          : Shape(ShapeType::CAPSULE)
          , radius(radius_)
          , length(length_)
        {
        }
        void createMsg(Message& msg) const override;

        float radius;
        float length;
    };

    class Mesh : public Shape
    {
      public:
        Mesh(ShapeType type = ShapeType::MESH)
          : Shape(type)
        {
        }
        Mesh(agx::Vec3Vector& vertices_, agx::UInt32Vector indices_);

        void createMsg(Message& msg) const override;

        agx::Vec3Vector vertices;
        agx::UInt32Vector indices;
    };

    class Line : public Shape
    {
      public:
        Line(const agx::Vec3& p1_ = agx::Vec3(), const agx::Vec3& p2_ = agx::Vec3())
          : Shape(ShapeType::LINE)
          , p1(p1_)
          , p2(p2_)
        {
        }

        void createMsg(Message& msg) const override;

        agx::Vec3 p1;
        agx::Vec3 p2;
    };

    class Cone : public Shape
    {
      public:
        Cone(float radius_ = 0.1f, float height_ = 1)
          : Shape(ShapeType::CONE)
          , radius(radius_)
          , height(height_)
        {
        }
        void createMsg(Message& msg) const override;

        float radius;
        float height;
    };

    class HollowCone : public Shape
    {
      public:
        HollowCone(
          float topRadius_ = 0.07f, float bottomRadius_ = 0.45f, float height_ = 0.9f, float thickness_ = 0.05f)
          : Shape(ShapeType::HOLLOW_CONE)
          , topRadius(topRadius_)
          , bottomRadius(bottomRadius_)
          , height(height_)
          , thickness(thickness_)
        {
        }

        void createMsg(Message& msg) const override;

        float topRadius;
        float bottomRadius;
        float height;
        float thickness;
    };

    class HollowCylinder : public Shape
    {
      public:
        HollowCylinder(float outerRadius_ = 0.07f, float height_ = 1, float thickness_ = 0.05f)
          : Shape(ShapeType::HOLLOW_CYLINDER)
          , outerRadius(outerRadius_)
          , height(height_)
          , thickness(thickness_)
        {
        }

        void createMsg(Message& msg) const override;

        float outerRadius;
        float height;
        float thickness;
    };


    class TruncatedCone : public Shape
    {
      public:
        TruncatedCone(float topRadius_ = 0.1f, float bottomRadius_ = 0.5f, float height_ = 1.0f)
          : Shape(ShapeType::TRUNCATED_CONE)
          , topRadius(topRadius_)
          , bottomRadius(bottomRadius_)
          , height(height_)
        {
        }

        void createMsg(Message& msg) const override;

        float topRadius;
        float bottomRadius;
        float height;
    };


    class HeightField : public Mesh
    {
      public:
        HeightField()
          : Mesh(ShapeType::HEIGHTFIELD)
        {
        }
        void createMsg(Message& msg) const override;
        std::pair<agx::UInt32Vector, agx::RealVector> modifiedVertices;
    };

    class Plane : public Shape
    {
      public:
        Plane()
          : Shape(ShapeType::PLANE)
        {
        }
        void createMsg(Message& msg) const override;
    };

    class Wire : public Shape
    {
      public:
        Wire()
          : Shape(ShapeType::WIRE)
        {
        }
        void createMsg(Message& msg) const override;
    };

    class Contacts : public Shape
    {
      public:
        Contacts()
          : Shape(ShapeType::CONTACTS)
        {
        }
        void createMsg(Message& msg) const override;
    };

    class RigidBodies : public Shape
    {
      public:
        RigidBodies()
          : Shape(ShapeType::RIGIDBODIES)
        {
        }
        void createMsg(Message& msg) const override;

        std::vector<double> positions;
    };

    class Granular : public Shape
    {
      public:
        Granular()
          : Shape(ShapeType::GRANULAR)
          , positions(nullptr)
          , radii(nullptr)
        {
        }


        void resize(size_t size)
        {
          position.resize(size * 3);
          radius.resize(size);
        }

        void createMsg(Message& msg) const override;

        agxData::Buffer* positions;
        agxData::Buffer* radii;

        std::vector<double> position;
        std::vector<float> radius;
    };

  }  // RemoteShapes

  #endif

  /**
  Implementation of a RenderProxyFactor that is resposible for sending data to a remote client whenever the 
  RenderManager tells the factory that a change has occured.
  */
  class AGXPHYSICS_EXPORT RemoteDebugRenderProxyFactory : public agxRender::RenderProxyFactory
  {
    public:

      /**
      Constructor
      \param port - Which port to which a client can connect.
      */
      RemoteDebugRenderProxyFactory(agx::UInt16 port = 9001);

      struct Statistics
      {
          Statistics()
            : packagesPerSecond(0)
            , bytesPerSecond(0)
          {
          }

          unsigned int packagesPerSecond;
          unsigned int bytesPerSecond;
      };

      /// \returns some statistics about amount of data being sent.
      Statistics getStatistics() const;

      /**
      \return the implementation of a GraphRenderer for remote rendering.
      */
      agxRender::Graph::GraphRenderer* getGraphRenderer() override;

      /**
      \param radius - The radius of the new sphere
      \return a pointer to a new SphereProxy with specified radius
      */
      agxRender::SphereProxy* createSphere(float radius) override;

      /**
      \param shape - The sphere that should be rendered
      \return a pointer to a new SphereProxy with specified radius
      */
      agxRender::SphereProxy* createSphere(agxCollide::Sphere* shape) override;

      /**
      \param text - The text
      \param pos - Position of the text. Currently only x,y is used.
      \param size - Size of text. Used to scale the actual text used.
      \return a pointer to a new TextProxy
      */
      agxRender::TextProxy* createText(const agx::String& text, const agx::Vec3& pos, float size = 1) override;

      /**
      \param p1, p2 - Start, end points in WORLD coordinate system
      \return a pointer to a new LineProxy
      */
      agxRender::LineProxy* createLine(const agx::Vec3& p1, const agx::Vec3& p2) override;

      /**
      \param shape - Line that should be rendered
      \return a pointer to a new LineProxy
      */
      agxRender::LineProxy* createLine(agxCollide::Line* shape) override;

      /**
      \param radius - The radius of a cylinder
      \param height - The height of a cylinder
      \return a pointer to a new CylinderProxy
      */
      agxRender::CylinderProxy* createCylinder(float radius, float height) override;

      /**
      \param shape - Cylinder that should be rendered
      \return a pointer to a new CylinderProxy
      */
      agxRender::CylinderProxy* createCylinder(agxCollide::Cylinder* shape) override;

      /**
      \param outerRadius - The outer radius for the a cylinder
      \param height - The height for the cylinder
      \param thickness - The thickness for the cylinder
      \return a pointer to a new HollowCylinderProxy
      */
      agxRender::HollowCylinderProxy* createHollowCylinder(float outerRadius, float height, float thickness) override;

      /**
      \param shape - HollowCylinder that should be rendered
      \return a pointer to a new HollowCylinderProxy
      */
      agxRender::HollowCylinderProxy* createHollowCylinder(agxCollide::HollowCylinder* shape) override;

      /**
      \param radius - The radius of a capsule
      \param height - The height of a capsule
      \return a pointer to a new CapsuleProxy
      */
      agxRender::CapsuleProxy* createCapsule(float radius, float height) override;

      /**
      \param shape - Capsule that should be renderered
      \return a pointer to a new CapsuleProxy
      */
      agxRender::CapsuleProxy* createCapsule(agxCollide::Capsule* shape) override;

      /**
      \param radius - The radius of a capsule
      \param height - The height of a capsule
      \param previousEndPoint0 - The previous lower end point of the WireShape.
      \param previousEndPoint1 - The previous upper end point of the WireShape.
      \return a pointer to a new WireShapeProxy
      */
      agxRender::WireShapeProxy* createWireShape(
        float radius, float height, const agx::Vec3& previousEndPoint0, const agx::Vec3& previousEndPoint1) override;

      /**
      \param halfExtents - The size of the box
      \return a pointer to a new BoxProxy with specified size
      */
      agxRender::BoxProxy* createBox(const agx::Vec3& halfExtents) override;

      /**
      \param shape - Box that should be rendered
      \return a pointer to a new BoxProxy
      */
      agxRender::BoxProxy* createBox(agxCollide::Box* shape) override;

      /**
      Create and return a RenderProxy for a Heightfield.
      \param shape - The heightfield for which a HeightfieldProxy will be created.
      */
      agxRender::HeightFieldProxy* createHeightfield(agxCollide::HeightField* shape) override;

      /**
      Create and return a RenderProxy for a Trimesh.
      \param shape - The mesh shape for which a TrimeshProxy will be created.
      */
      agxRender::TrimeshProxy* createTrimesh(agxCollide::Trimesh* shape) override;

      /**
      \param radius - The radius of the cone
      \param height - The height of the cone
      \return a pointer to a new ConeProxy
      */
      agxRender::ConeProxy* createCone(float radius, float height) override;

      /**
      \param shape - Cone that should be rendered
      \return a pointer to a new TruncatedConeProxy
      */
      agxRender::TruncatedConeProxy* createCone(agxCollide::Cone* shape) override;

      /**
      \param topRadius - The top radius of the cone
      \param bottomRadius - The bottom radius of the cone
      \param height - The height of the cone
      \return a pointer to a new ConeProxy
      */
      agxRender::TruncatedConeProxy* createTruncatedCone(float topRadius, float bottomRadius, float height) override;

      /**
      \param topRadius - The top radius of the cone
      \param bottomRadius - The bottom radius of the cone
      \param height - The height of the cone
      \param thickness - The thickness of the cone
      */
      agxRender::HollowConeProxy* createHollowCone(
        float topRadius, float bottomRadius, float height, float thickness) override;

      /**
      \param shape - HollowCone that should be rendered
      \return a pointer to a new HollowConeProxy
      */
      agxRender::HollowConeProxy* createHollowCone(agxCollide::HollowCone* shape) override;

      /**
      \param normal - The normal of a plane
      \param distance - The scalar part of the plane
      \return a pointer to a new PlaneProxy
      */
      agxRender::PlaneProxy* createPlane(const agx::Vec3& normal, agx::Real distance) override;

      /**
      \param shape - The plane that should be rendered
      \return a pointer to a new PlaneProxy
      */
      agxRender::PlaneProxy* createPlane(agxCollide::Plane* shape) override;

      /**
      \param contacts - Vector with all contacts that should be visualized
      \return a pointer to a new ContactsProxy
      */
      agxRender::ContactsProxy* createContacts(const agxCollide::GeometryContactPtrVector& contacts) override;

      agxRender::RigidBodyBatchRenderProxy* createRigidBodies(const agx::RigidBodyPtrSetVector* enabledBodies) override;

      agxRender::WireRenderProxy* createWire(float radius, const agx::Vec3& color) override;


      agxRender::RenderProxy* createSphereBatchRenderer(
        agxData::Buffer* positions, agxData::Buffer* rotations, agxData::Buffer* radii, agxData::Buffer* colors,
        agxData::Buffer* enableRendering, agxData::Value* bound, agx::Component* context) override;


      /**
      Set a transformation which will be applied to all transformations for RenderProxy's.
      This can be used to transform all objects to the center of the earth instead of trying to render on the surface of
      the earth (6000km). This transform is by default I. If you for example do m.setTranslate(-10,0,0) all
      RenderProxy's will be rendered translated -10 units in X. If a RenderManager is present, RenderManager::update()
      will be called
      */
      void setGlobalTransform(const agx::AffineMatrix4x4& m) override;

      /**
      \return the current global transformation
      */
      const agx::AffineMatrix4x4& getGlobalTransform() const { return m_invTransform; }

      /**
      Set the RenderMode for a specified node
      */
      // void setRenderMode(agxRender::RenderProxy* proxy, agxRender::RenderProxy::RenderMode mode);


      // Methods that operate on the remote representation
      void updateShape(unsigned int proxyID);
      void setTransform(unsigned int proxyID, const agx::AffineMatrix4x4& transform);
      void setColor(unsigned int proxyID, const agx::Vec3& color);
      void setEnable(unsigned int proxyID, bool flag);
      void remove(unsigned int proxyID);

      agx::ref_ptr<agx::Referenced> getProxyStorage();

    protected:
      void setRenderManager(agxRender::RenderManager* mgr) override;

      agx::ref_ptr<agxSDK::EventListener> m_updateListener;

      agx::ref_ptr<agx::Referenced> m_proxyStorage;

    public:
      /// Class for storing specific data for each RenderProxy
      template <typename T>
      struct ShapeData
      {
          ShapeData() {}
          ShapeData(const ShapeData& other)
          {
            shape = other.shape;
            material = other.material;
          }

          RemoteShapes::Shape* getShape() { return &shape; }
          const RemoteShapes::Shape* getShape() const { return &shape; }

          T shape;
          agxSDK::RemoteShapes::Material material;
      };


    protected:
      /// Destructor
      virtual ~RemoteDebugRenderProxyFactory();


      // agx::ref_ptr<GraphRenderer> m_graphRenderer;
      agx::AffineMatrix4x4 m_invTransform;

      agx::UInt16 m_port;
  };

  #ifndef SWIG


  struct RemoteData_Base
  {
    static unsigned int m_proxyIDCounter;
  };

  /// Class for handling methods related to remote rendering such as setColor, setTransform etc.
  template <typename T>
  class RemoteData : public RemoteData_Base
  {
    public:
      RemoteData(
        const RemoteDebugRenderProxyFactory::ShapeData<T>& data, RemoteDebugRenderProxyFactory* factory,
        agxRender::RenderProxy* proxy)
        : m_data(data)
        , m_factory(factory)
        , m_cachedTransform(agx::Quat(), agx::Vec3(-agx::Infinity))
        , m_proxy(proxy)
        , m_proxyID(m_proxyIDCounter++)
      {
      }

      RemoteData(RemoteDebugRenderProxyFactory* factory, agxRender::RenderProxy* proxy)
        : m_data()
        , m_factory(factory)
        , m_cachedTransform(agx::Quat(), agx::Vec3(-agx::Infinity))
        , m_proxy(proxy)
        , m_proxyID(m_proxyIDCounter++)
      {
        setColorRemote(agxRender::Color::Red(), 1);
      }

      unsigned int getProxyID() const { return m_proxyID; }

      /// Transform a node
      void setTransformRemote(const agx::AffineMatrix4x4& transform)
      {
        /// Take the global transform into account
        agx::AffineMatrix4x4 m = transform * m_factory->getGlobalTransform();

        if (agx::equivalent(transform, m_cachedTransform))
          return;

        m_cachedTransform = transform;

        m_factory->setTransform(getProxyID(), m);
      }

      /// Enable/disable rendering
      void updateShapeRemote() { m_factory->updateShape(getProxyID()); }


      /// Enable/disable rendering
      void setEnableRemote(bool flag)
      {
        if (!flag) {  // Remove node from all its parents
          m_factory->setEnable(getProxyID(), false);
        }
        else {  // Add it for rendering again.
          m_factory->setEnable(getProxyID(), true);
        }
      }

      /// Set the alpha value
      void setAlphaRemote(float transparency)
      {
        transparency = 0;
        // We do not support changing alpha value for now.
        // m_data.material->setTransparency(Remote::Material::FRONT_AND_BACK, 1 - transparency);
      }

      /// Set the color
      void setColorRemote(const agx::Vec3& color, float /* alpha*/)
      {
        agxRender::Color col = m_data.material.color;

        if (agx::equivalent(agx::Vec3f(color), col.asVec3()))
          return;

        m_data.material.color = agxRender::Color((float)color[0], (float)color[1], (float)color[2], 1);

        // Send color
        m_factory->setColor(getProxyID(), color);
      }

      /// Remove the node from all its parents
      void onRemoveRemote() { m_factory->remove(getProxyID()); }

      /// Change the render mode
      void setRenderModeRemote(agxRender::RenderProxy::RenderMode mode)
      {
        // Send RenderMode
        mode = agxRender::RenderProxy::WIREFRAME;
      }

      const RemoteDebugRenderProxyFactory::ShapeData<T>* getData() const { return &m_data; }
      RemoteDebugRenderProxyFactory::ShapeData<T>* getData() { return &m_data; }

    protected:
      virtual ~RemoteData() {}

      RemoteDebugRenderProxyFactory::ShapeData<T> m_data;
      RemoteDebugRenderProxyFactory* m_factory;
      agx::AffineMatrix4x4 m_invTransform;
      agx::AffineMatrix4x4 m_cachedTransform;
      agxRender::RenderProxy* m_proxy;
      unsigned int m_proxyID;
  };

    #ifndef SWIG

      #ifdef ADD_COMMON_REMOTE_PROXY_METHODS
        #undef ADD_COMMON_REMOTE_PROXY_METHODS
      #endif

      /// Macro for adding some methods for each specialization of RenderProxy
      #define ADD_COMMON_REMOTE_PROXY_METHODS()                         \
        void onChange(RenderProxy::EventType type) override             \
        {                                                               \
          switch (type) {                                               \
            case (RenderProxy::ENABLE):                                 \
              this->setEnableRemote(this->getEnable());                 \
              break;                                                    \
            case (RenderProxy::ALPHA):                                  \
              this->setAlphaRemote(this->getAlpha());                   \
              break;                                                    \
            case (RenderProxy::TRANSFORM):                              \
              this->setTransformRemote(this->getTransform());           \
              break;                                                    \
            case (RenderProxy::COLOR):                                  \
              this->setColorRemote(this->getColor(), this->getAlpha()); \
              break;                                                    \
            case (RenderProxy::SHAPE):                                  \
              this->updateShapeRemote();                                \
              break;                                                    \
            case (RenderProxy::REMOVE):                                 \
              this->onRemoveRemote();                                   \
              break;                                                    \
            case (RenderProxy::RENDERMODE):                             \
              this->setRenderModeRemote(this->getRenderMode());         \
              break;                                                    \
          }                                                             \
        }                                                               \
        virtual bool updateShape() override
    #endif



  /// Implementation of remote-rendered SphereProxy
  class AGXPHYSICS_EXPORT SphereProxy
    : public agxRender::SphereProxy
    , public RemoteData<RemoteShapes::Sphere>
  {
    public:
      SphereProxy(
        float radius, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Sphere> sphere,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~SphereProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered TextProxy
  class AGXPHYSICS_EXPORT TextProxy
    : public agxRender::TextProxy
    , public RemoteData<RemoteShapes::Text>
  {
    public:
      TextProxy(
        const agx::String& text, const agx::Vec3& pos,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Text> data, RemoteDebugRenderProxyFactory* factory);


    protected:
      virtual ~TextProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };


  /// Implementation of remote-rendered BoxProxy
  class AGXPHYSICS_EXPORT BoxProxy
    : public agxRender::BoxProxy
    , public RemoteData<RemoteShapes::Box>
  {
    public:
      BoxProxy(
        const agx::Vec3& halfExtents, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Box> box,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~BoxProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered LineProxy
  class AGXPHYSICS_EXPORT LineProxy
    : public agxRender::LineProxy
    , public RemoteData<RemoteShapes::Line>
  {
    public:
      LineProxy(
        const agx::Vec3& p1, const agx::Vec3& p2, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Line> line,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~LineProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered CylinderProxy
  class AGXPHYSICS_EXPORT CylinderProxy
    : public agxRender::CylinderProxy
    , public RemoteData<RemoteShapes::Cylinder>
  {
    public:
      CylinderProxy(
        float radius, float height, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Cylinder> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~CylinderProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered HollowCylinderProxy
  class AGXPHYSICS_EXPORT HollowCylinderProxy
    : public agxRender::HollowCylinderProxy
    , public RemoteData<RemoteShapes::HollowCylinder>
  {
    public:
      HollowCylinderProxy(
        float radius, float height, float thickness,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::HollowCylinder> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~HollowCylinderProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered ConeProxy
  class AGXPHYSICS_EXPORT ConeProxy
    : public agxRender::ConeProxy
    , public RemoteData<RemoteShapes::Cone>
  {
    public:
      ConeProxy(
        float radius, float height, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Cone> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~ConeProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered TruncatedConeProxy
  class AGXPHYSICS_EXPORT TruncatedConeProxy
    : public agxRender::TruncatedConeProxy
    , public RemoteData<RemoteShapes::TruncatedCone>
  {
    public:
      TruncatedConeProxy(
        float topRadius, float bottomRadius, float height,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::TruncatedCone> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~TruncatedConeProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered HollowConeProxy
  class AGXPHYSICS_EXPORT HollowConeProxy
    : public agxRender::HollowConeProxy
    , public RemoteData<RemoteShapes::HollowCone>
  {
    public:
      HollowConeProxy(
        float topRadius, float bottomRadius, float height, float thickness,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::HollowCone> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~HollowConeProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered PlaneProxy
  class AGXPHYSICS_EXPORT PlaneProxy
    : public agxRender::PlaneProxy
    , public RemoteData<RemoteShapes::Plane>
  {
    public:
      PlaneProxy(
        const agx::Vec3& normal, agx::Real distance, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Plane> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~PlaneProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered CapsuleProxy
  class AGXPHYSICS_EXPORT CapsuleProxy
    : public agxRender::CapsuleProxy
    , public RemoteData<RemoteShapes::Capsule>
  {
    public:
      CapsuleProxy(
        float radius, float height, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Capsule> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~CapsuleProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };


  /// Implementation of remote-rendered WireShapeProxy
  class AGXPHYSICS_EXPORT WireShapeProxy
    : public agxRender::WireShapeProxy
    , public RemoteData<RemoteShapes::Wire>
  {
    public:
      WireShapeProxy(
        float radius, float height, const agx::Vec3& previousEndPoint0, const agx::Vec3& previousEndPoint1,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Wire> data, RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~WireShapeProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered HeightfieldProxy
  class AGXPHYSICS_EXPORT HeightFieldProxy
    : public agxRender::HeightFieldProxy
    , public RemoteData<RemoteShapes::HeightField>
  {
    public:
      HeightFieldProxy(
        agxCollide::HeightField* hf, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::HeightField> hfData,
        RemoteDebugRenderProxyFactory* factory);

      void set(const agx::Vec2iVector& modifiedIndices, const agx::RealVector& heights) override;

    protected:
      virtual ~HeightFieldProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };

  /// Implementation of remote-rendered TrimeshProxy
  class AGXPHYSICS_EXPORT TrimeshProxy
    : public agxRender::TrimeshProxy
    , public RemoteData<RemoteShapes::Mesh>
  {
    public:
      TrimeshProxy(
        agxCollide::Trimesh* mesh, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Mesh> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~TrimeshProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();
  };


  class ContactsProxy
    : public agxRender::ContactsProxy
    , public RemoteData<RemoteShapes::Contacts>
  {
    public:
      ContactsProxy(
        const agxCollide::GeometryContactPtrVector& contacts,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Contacts> data, RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~ContactsProxy() {}

      ADD_COMMON_REMOTE_PROXY_METHODS();

      virtual void reset() override
      {
        // I don't think we can or could do anything during reset.
      }
  };

  class WireRenderProxy
    : public agxRender::WireRenderProxy
    , public RemoteData<RemoteShapes::Wire>
  {
    public:
      WireRenderProxy(
        float radius, const agx::Vec3& color, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Wire> data,
        RemoteDebugRenderProxyFactory* factory);

    protected:
      virtual ~WireRenderProxy();

      ADD_COMMON_REMOTE_PROXY_METHODS();

      /**
      Called when the wire is added to the simulation.
      */
      virtual void onAddNotification();

      /**
      Called when the wire is removed from the simulation.
      */
      virtual void onRemoveNotification();

      virtual void reset() override
      {
        // I don't think we can or could do anything during reset.
      }
  };


  class RigidBodyBatchRenderProxy
    : public agxRender::RigidBodyBatchRenderProxy
    , public RemoteData<RemoteShapes::RigidBodies>
  {
    public:
      RigidBodyBatchRenderProxy(
        const agx::RigidBodyPtrSetVector* enabledBodies,
        RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::RigidBodies> data,
        RemoteDebugRenderProxyFactory* factory);


    protected:
      virtual ~RigidBodyBatchRenderProxy();

      // This is virtual void updateShape().
      ADD_COMMON_REMOTE_PROXY_METHODS();


      virtual void reset() override
      {
        // I don't think we can or could do anything during reset.
      }
  };

  class GranularRenderProxy
    : public agxRender::RenderProxy
    , public RemoteData<RemoteShapes::Granular>
  {
    public:
      GranularRenderProxy(
        agxData::Buffer* positions, agxData::Buffer* rotations, agxData::Buffer* radii, agxData::Buffer* colors,
        agxData::Buffer* enableRendering, RemoteDebugRenderProxyFactory::ShapeData<RemoteShapes::Granular> data,
        RemoteDebugRenderProxyFactory* factory);


    protected:
      virtual ~GranularRenderProxy();

      // This is virtual void updateShape().
      ADD_COMMON_REMOTE_PROXY_METHODS();


      virtual void reset() override
      {
        // I don't think we can or could do anything during reset.
      }
  };

  #endif  // SWIG

  #ifdef ADD_COMMON_REMOTE_PROXY_METHODS
    #undef ADD_COMMON_REMOTE_PROXY_METHODS
  #endif

}

#endif
